#!/usr/bin/env python3
from __future__ import annotations

import sys
import time

import numpy as np


class MazeProblem:
    """Represents data for instance of maze problem."""

    def __init__(self, width: int, height: int, step_reward: float, delay_reward, goal_reward: float, gamma: float,
                 epsilon: float, layout: list[list[str]]):
        self.width: int = width
        """Width of maze"""
        self.height: int = height
        """Height of maze"""
        self.step_reward: float = step_reward
        """Reward for taking a step onto an empty tile"""
        self.delay_reward: float = delay_reward
        """Reward for taking a step onto an delay tile"""
        self.goal_reward: float = goal_reward
        """Reward for taking a step onto an goal tile"""
        self.gamma: float = gamma
        """Discount factor for future rewards"""
        self.epsilon: float = epsilon
        """Stopping criteria for the value iteration algorithm, used to determine when the algorithm should terminate"""
        self.layout: list[list[str]] = layout
        """Stores structure of the maze. Used row major order. Value of layout[i][j] corresponds to i-th row
        and j-th position in row. Range of expected values should be {' ', '#', 'D', 'G'}."""

    def __str__(self):
        string = f"{self.width} {self.height} {self.step_reward} {self.delay_reward} {self.goal_reward} {self.gamma} " \
                 f"{self.epsilon}\n"
        for row in self.layout:
            string += f"{''.join(row)}\n"
        return string

    @staticmethod
    def load_from_file(file: str) -> MazeProblem:
        with open(file, 'r') as f:
            width, height, step_reward, delay_reward, goal_reward, gamma, epsilon = map(float, f.readline().split())
            lines = f.readlines()
        layout = [list(line.replace("\n", "")) for line in lines]
        return MazeProblem(int(width), int(height), step_reward, delay_reward, goal_reward, gamma, epsilon, layout)


def main(instance_file, solution_file):
    instance = MazeProblem.load_from_file(instance_file)

    start_time = time.time()

    # todo implement VI algorithm
    vi_values_result = np.zeros((instance.width, instance.height))

    elapsed_time = time.time() - start_time
    print(f"Elapsed VI time: {elapsed_time:.3f} seconds")

    with open(solution_file, 'w') as f:
        for row in vi_values_result:
            f.write(' '.join(map(str, row)) + '\n')


if __name__ == "__main__":
    instance_path = sys.argv[1]
    solution_path = sys.argv[2]
    main(instance_path, solution_path)
