#include <queue>
#include <mutex>
#include <condition_variable>
#include <thread>
#include <iostream>
#include <chrono>

template <typename T>
class blocking_bounded_queue {
public:
    blocking_bounded_queue(size_t max_size = 10):
        max_size(max_size)
    {}

    bool add(T const& e) {
        std::unique_lock<std::mutex> lg(mutex);
        add_var.wait(lg, [&] () -> bool { return queue.size() < max_size || closing; });
        if (closing) {
            return false;
        }
        queue.push(e);
        take_var.notify_one();
        return true;
    }

    std::pair<bool, T> take() {
        std::unique_lock<std::mutex> lg(mutex);
        take_var.wait(lg, [&] () -> bool { return queue.size() > 0 || closing; });
        if (queue.size() == 0) {
            return { false, T{} };
        }
        auto out = std::move(queue.front()); queue.pop();
        add_var.notify_one();
        return { true, std::move(out) };
    }

    void close() {
        std::unique_lock<std::mutex> lg(mutex);
        closing = true;
        take_var.notify_all();
        add_var.notify_all();
    }


private:
    std::queue<T> queue;
    std::mutex mutex;
    std::condition_variable take_var, add_var;
    size_t max_size;
    bool closing = false;
};

int main() {
    using namespace std::chrono_literals;
    blocking_bounded_queue<int> queue;
    auto producer = [] (blocking_bounded_queue<int>& queue) {
        for (int i = 0; i < 100; ++i) {
            queue.add(i);
            std::cout << "Added element\n";
        }
        queue.close();
    };

    auto consumer = [] (blocking_bounded_queue<int>& queue) {
        while (true) {
            auto elem = queue.take();
            if (!elem.first) {
                break;
            }
            std::cout << elem.second << '\n';
            std::this_thread::sleep_for(100ms);
        }
    };

    std::thread t1(producer, std::ref(queue));
    std::thread t2(consumer, std::ref(queue));

    t1.join();
    t2.join();
}
