#pragma once

#include "CoreMinimal.h"
#include "ClothSimulationComponent.h"
#include "RealtimeMeshActor.h"
#include "RealtimeMeshSimple.h"
#include "ClothActor.generated.h"

UCLASS()
class CV06_API AClothActor : public ARealtimeMeshActor
{
	GENERATED_BODY()
	
public:	
	// Sets default values for this actor's properties
	AClothActor();

	// These parameters are passed to the actual simulation component.
	// Mass of each cloth node.
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category="Cloth Simulation")
	float NodeMass = 0.1f;

	// Elastic modulus of the structural springs.
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category="Cloth Simulation")
	float StructuralElasticity = 1.0f;

	// Elastic modulus of the shear springs.
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category="Cloth Simulation")
	float ShearElasticity = 1.0f;

	// Elastic modulus of the flexion springs.
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category="Cloth Simulation")
	float BendElasticity = 1.0f;

	// Damping factor - higher damping means higher viscosity.
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category="Cloth Simulation")
	float Damping = 1.0f;

	// Slows down the time step by multiplying DeltaTime by this value. This
	// is passed to the simulation every tick.
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category="Cloth Simulation")
	float TimeScale = 1.0f;

	// Whether the simulation uses a separate timer and time step, to avoid
	// instabilities caused by too large a time step.
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category="Cloth Simulation")
	bool bUseConstantTimeStep = false;

	// Amount of time to advance when using constant time step.
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category="Cloth Simulation")
	float ConstantTimeStep = 1.0f / 90.0f;

	// Scale of the generated cloth.
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category="Cloth Simulation")
	float Scale = 100.0f;

	// Number of squares in the grid (both width and height).
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category="Cloth Simulation")
	int32 GridSize = 10;

	// Additional wind force to apply to the cloth.
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="Cloth Simulation")
	FVector WindForce = FVector::ZeroVector;

	// Material to use on the generated mesh.
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category="Cloth Display")
	UMaterialInterface* Material;

protected:
	// Called when the game starts or when spawned
	virtual void BeginPlay() override;

private:
	// Generate a grid for RuntimeMeshComponent. Creates vertices and their parameters, and also
	// creates triangle indices. This is saved to MeshData for later use.
	void GenerateGrid();

	// Prepares RuntimeMeshComponent for first data upload, generates the mesh grid using GenerateGrid
	// and uploads the data to RMC.
	void GenerateMesh();

	// Takes current simulation node positions and updates the grid mesh accordingly.
	void UpdateMesh();

public:	
	// Called every frame
	virtual void Tick(float DeltaTime) override;

	UPROPERTY()
	TObjectPtr<UClothSimulationComponent> ClothSimulation;

	UPROPERTY()
	TObjectPtr<URealtimeMeshSimple> Mesh;

	RealtimeMesh::FRealtimeMeshStreamSet MeshData;
	//FRealtimeMeshSimpleMeshData MeshData;
	FRealtimeMeshSectionGroupKey MeshSectionKey;
};
