#include "ClothActor.h"

#include "RealtimeMeshSimple.h"


// Sets default values
AClothActor::AClothActor()
{
 	// Set this actor to call Tick() every frame.  You can turn this off to improve performance if you don't need it.
	PrimaryActorTick.bCanEverTick = true;

	ClothSimulation = CreateDefaultSubobject<UClothSimulationComponent>(TEXT("Cloth Simulation"));
}

// Called when the game starts or when spawned
void AClothActor::BeginPlay()
{
	Super::BeginPlay();
	
	ClothSimulation->NodeMass = NodeMass;
	ClothSimulation->StructuralElasticity = StructuralElasticity;
	ClothSimulation->ShearElasticity = ShearElasticity;
	ClothSimulation->BendElasticity = BendElasticity;
	ClothSimulation->Damping = Damping;
	ClothSimulation->TimeScale = TimeScale;
	ClothSimulation->Scale = Scale;
	ClothSimulation->GridSize = GridSize;
	ClothSimulation->bUseConstantTimeStep = bUseConstantTimeStep;
	ClothSimulation->ConstantTimeStep = ConstantTimeStep;
	ClothSimulation->WindForce = WindForce;
	ClothSimulation->GenerateCloth();

	if (bUseConstantTimeStep)
	{
		ClothSimulation->StartConstantTimer();
	}

	GenerateMesh();
}

void AClothActor::GenerateGrid()
{
	RealtimeMesh::TRealtimeMeshBuilderLocal<> Builder(MeshData);
	Builder.EnableTangents();
	Builder.EnableColors();
	Builder.EnableTexCoords();
	
	int32 VertexIndex = 0;
	for (int32 Y = 0; Y < GridSize; ++Y)
	{
		for (int32 X = 0; X < GridSize; ++X, ++VertexIndex)
		{
			// Each vertex gets a position from the simulation and a texture coordinate.
			FVector3f Position = FVector3f(ClothSimulation->GetNodePosition(VertexIndex));

			float U = X / static_cast<float>(GridSize);
			float V = Y / static_cast<float>(GridSize);

			// Add the new vertex with a white color and set texture coords.
			Builder.AddVertex(Position)
				.SetNormalAndTangent(FVector3f::ZeroVector, FVector3f::ZeroVector)
				.SetTexCoord({U, V})
				.SetColor(FColor::White);

			// When we have enough vertices and are not at the start of a line, we can start
			// generating triangles from them. Each vertex adds 2 triangles to the grid (1 quad).
			if (X > 0 && Y > 0)
			{
				const uint32 TopRightIndex = VertexIndex - GridSize;
				const uint32 TopLeftIndex = VertexIndex - GridSize - 1;
				const uint32 BottomLeftIndex = VertexIndex - 1;
				const uint32 BottomRightIndex = VertexIndex;
				
				// Now create two triangles from these four vertices.
				// The order of these (clockwise/counter-clockwise) dictates which way the triangle will face.
				Builder.AddTriangle(BottomLeftIndex, TopRightIndex, TopLeftIndex);
				Builder.AddTriangle(BottomLeftIndex, BottomRightIndex, TopRightIndex);

				const FVector3f NormalCurrent = FVector3f::CrossProduct(
					Builder.GetPosition(BottomLeftIndex) - Builder.GetPosition(TopLeftIndex),
					(Builder.GetPosition(TopLeftIndex) - Builder.GetPosition(TopRightIndex)).GetSafeNormal());

				Builder.SetNormal(TopLeftIndex, NormalCurrent);
				Builder.SetNormal(TopRightIndex, NormalCurrent);
				Builder.SetNormal(BottomLeftIndex, NormalCurrent);
				Builder.SetNormal(BottomRightIndex, NormalCurrent);
			}
		}
	}
}

void AClothActor::GenerateMesh()
{
	Mesh = RealtimeMeshComponent->InitializeRealtimeMesh<URealtimeMeshSimple>();
	Mesh->SetupMaterialSlot(0, TEXT("Cloth Material"), Material);

	MeshSectionKey = FRealtimeMeshSectionGroupKey::Create(0, TEXT(""));
	GenerateGrid();
	
	Mesh->CreateSectionGroup(MeshSectionKey, MeshData);
}

void AClothActor::UpdateMesh()
{
	auto Positions = MeshData.Find(RealtimeMesh::FRealtimeMeshStreams::Position);
	
	int32 VertexIndex = 0;
	for (int32 Y = 0; Y < GridSize; ++Y)
	{
		for (int32 X = 0; X < GridSize; ++X, ++VertexIndex)
		{
			// Each vertex gets its position updated from the simulation.
			FVector3f Position = FVector3f(ClothSimulation->GetNodePosition(VertexIndex));
			*Positions->GetDataAtVertex<FVector3f>(VertexIndex) = Position;
		}
	}
	Mesh->UpdateSectionGroup(MeshSectionKey, MeshData);
}

// Called every frame
void AClothActor::Tick(float DeltaTime)
{
	Super::Tick(DeltaTime);
	
	ClothSimulation->TimeScale = TimeScale;

	UpdateMesh();
}

