// Adapted from: Jos Stam, "Real-Time Fluid Dynamics for Games". Proceedings of the Game Developer Conference, March 2003.
// For details on how the individual functions work, please consult the above publication.

#include "StamFluidBox.h"

#include "Engine/VolumeTexture.h"

DEFINE_LOG_CATEGORY(StamFluidBox);

// Sets default values
AStamFluidBox::AStamFluidBox()
{
 	// Set this actor to call Tick() every frame.  You can turn this off to improve performance if you don't need it.
	PrimaryActorTick.bCanEverTick = true;
	PrimaryActorTick.TickGroup = ETickingGroup::TG_DuringPhysics;

	StaticMeshComponent = CreateDefaultSubobject<UStaticMeshComponent>(TEXT("Fluid Box"));
	StaticMeshComponent->SetGenerateOverlapEvents(true);
	StaticMeshComponent->SetCollisionObjectType(ECC_GameTraceChannel1);
	StaticMeshComponent->SetCollisionProfileName(TEXT("OverlapAllDynamic"));
	RootComponent = StaticMeshComponent;

	auto MeshFinder = ConstructorHelpers::FObjectFinder<UStaticMesh>(TEXT("/Script/Engine.StaticMesh'/Engine/BasicShapes/Cube.Cube'"));
	if (MeshFinder.Succeeded())
	{
		StaticMeshComponent->SetStaticMesh(MeshFinder.Object);
	}

	auto MaterialFinder = ConstructorHelpers::FObjectFinder<UMaterial>(TEXT("/Script/Engine.Material'/Game/Materials/M_Volume.M_Volume'"));
	if (MaterialFinder.Succeeded())
	{
		Material = MaterialFinder.Object;
	}
}

// Called when the game starts or when spawned
void AStamFluidBox::BeginPlay()
{
	Super::BeginPlay();

	CreateVolumeTransient();
	TObjectPtr<UMaterialInstanceDynamic> MaterialInstance = UMaterialInstanceDynamic::Create(Material, StaticMeshComponent);
	MaterialInstance->SetTextureParameterValue(TEXT("Volume"), VolumeTexture);
	StaticMeshComponent->SetMaterial(0, MaterialInstance);

	DensityGridSize = {GridSize.X + 2, GridSize.Y + 2, GridSize.Z + 2};
	const int Total = DensityGridSize.X * DensityGridSize.Y * DensityGridSize.Z;
	
	Density.Init(0, Total);
	DensityPrev.Init(0, Total);

	VelocityX.Init(0, Total);
	VelocityXPrev.Init(0, Total);
	
	VelocityY.Init(0, Total);
	VelocityYPrev.Init(0, Total);
	
	VelocityZ.Init(0, Total);
	VelocityZPrev.Init(0, Total);
}

// Called every frame
void AStamFluidBox::Tick(float DeltaTime)
{
	Super::Tick(DeltaTime);
	
	// Run Stam's stable fluid solver
	if (bSimulating)
	{
		VelocityStep(DeltaTime);
		DensityStep(DeltaTime);
	}

	// Log output for testing
	//UE_LOG(StamFluidBox, Warning, TEXT("Value at center: %f"), Density[CellIndex(GridSize.X / 2 + 1, GridSize.Y / 2 + 1, GridSize.Z / 2 + 1)]);

	// Lock texture data so we can update it
	uint8* ByteArray = static_cast<uint8*>(VolumeTexture->GetPlatformData()->Mips[0].BulkData.Lock(LOCK_READ_WRITE));

	// Loop through all our cells and convert their values to color.
	// Only takes into account values between 0.0-1.0 and maps them to 0-255
	int Current = 0;
	for (int i = 1; i <= GridSize.Z; i++)
	{
		for (int j = 1; j <= GridSize.Y; j++)
		{
			for (int k = 1; k <= GridSize.X; k++)
			{
				const float Scale = FMath::Clamp(Density[CellIndex(k, j, i)], 0.0f, 1.0f);

				// 4 values: R G B A, a byte each
				ByteArray[Current + 0] = static_cast<int8>(Scale * FluidColor.R);
				ByteArray[Current + 1] = static_cast<int8>(Scale * FluidColor.G);
				ByteArray[Current + 2] = static_cast<int8>(Scale * FluidColor.B);
				ByteArray[Current + 3] = static_cast<int8>(Scale * FluidColor.A);

				Current += 4;
			}
		}
	}

	// We're done updating - unlock the data and tell UE to update the texture on the GPU side.
	VolumeTexture->GetPlatformData()->Mips[0].BulkData.Unlock();
	VolumeTexture->UpdateResource();

	// Clear data for next tick.
	const int N = DensityGridSize.X * DensityGridSize.Y * DensityGridSize.Z;
	for (int i = 0; i < N; ++i)
	{
		DensityPrev[i] = VelocityXPrev[i] = VelocityYPrev[i] = VelocityZPrev[i] = 0.0f;
	}
}

void AStamFluidBox::AddFluidAtLocation(FVector WorldLocation, float Amount)
{
	FVector LocalLocation = GetTransform().Inverse().TransformPosition(WorldLocation);
	
	LocalLocation.X = FMath::Clamp((LocalLocation.X + 50) / 100.0f * GridSize.X, 0, GridSize.X);
	LocalLocation.Y = FMath::Clamp((LocalLocation.Y + 50) / 100.0f * GridSize.Y, 0, GridSize.Y);
	LocalLocation.Z = FMath::Clamp((LocalLocation.Z + 50) / 100.0f * GridSize.Z, 0, GridSize.Z);
	
	FIntVector GridLocation = {
		static_cast<int>(LocalLocation.X),
		static_cast<int>(LocalLocation.Y),
		static_cast<int>(LocalLocation.Z),
	};
	DensityPrev[CellIndex(GridLocation)] += Amount;
}

void AStamFluidBox::AddForceAtLocation(FVector WorldLocation, FVector Force)
{
	FTransform InverseTransform = GetTransform().Inverse();
	FVector LocalLocation = InverseTransform.TransformPosition(WorldLocation);
	FVector LocalForce = InverseTransform.TransformVector(Force);
	
	LocalLocation.X = FMath::Clamp((LocalLocation.X + 50) / 100.0f * GridSize.X, 0, GridSize.X);
	LocalLocation.Y = FMath::Clamp((LocalLocation.Y + 50) / 100.0f * GridSize.Y, 0, GridSize.Y);
	LocalLocation.Z = FMath::Clamp((LocalLocation.Z + 50) / 100.0f * GridSize.Z, 0, GridSize.Z);
	
	FIntVector GridLocation = {
		static_cast<int>(LocalLocation.X),
		static_cast<int>(LocalLocation.Y),
		static_cast<int>(LocalLocation.Z),
	};
	VelocityXPrev[CellIndex(GridLocation)] += LocalForce.X;
	VelocityYPrev[CellIndex(GridLocation)] += LocalForce.Y;
	VelocityZPrev[CellIndex(GridLocation)] += LocalForce.Z;
}

void AStamFluidBox::CreateVolumeTransient()
{
	// First create the object itself.
	VolumeTexture = NewObject<UVolumeTexture>(GetTransientPackage(), NAME_None, RF_Transient);

	FTexturePlatformData* PlatformData = VolumeTexture->GetPlatformData();
	// Newly created Volume textures have this null'd
	if (!PlatformData)
	{
		VolumeTexture->SetPlatformData(new FTexturePlatformData());
		PlatformData = VolumeTexture->GetPlatformData();
	}
	// Set Dimensions and Pixel format.
	PlatformData->SizeX = GridSize.X;
	PlatformData->SizeY = GridSize.Y;
	PlatformData->SetNumSlices(GridSize.Z);
	PlatformData->PixelFormat = EPixelFormat::PF_R8G8B8A8;
	// Set sRGB and streaming to false.
	VolumeTexture->SRGB = false;
	VolumeTexture->NeverStream = true;

	constexpr int PixelByteSize = 8 * 4;
	const int64 TotalSize = GridSize.X * GridSize.Y * GridSize.Z * PixelByteSize;

	// Create the one and only mip in this texture.
	FTexture2DMipMap* Mip = new FTexture2DMipMap();
	Mip->SizeX = GridSize.X;
	Mip->SizeY = GridSize.Y;
	Mip->SizeZ = GridSize.Z;

	Mip->BulkData.Lock(LOCK_READ_WRITE);
	// Allocate memory in the mip and copy the actual texture data inside
	uint8* ByteArray = Mip->BulkData.Realloc(TotalSize);

	// Memset to zero
	FMemory::Memset(ByteArray, 0, TotalSize);

	Mip->BulkData.Unlock();
	
	// Add the new MIP to the list of mips.
	PlatformData->Mips.Add(Mip);

	// Tell Unreal to update the rest.
	VolumeTexture->UpdateResource();
}

void AStamFluidBox::VelocityStep(float DeltaTime)
{
	// Stam's original code uses pointer swapping - here we just swap the variable names instead.
	// Each vector component is processed separately as per Stam's original algorithm. We could also
	// use Unreal's vectors for this, or use this separation for parallel computation.
	AddSources(VelocityX, VelocityXPrev, DeltaTime);
	AddSources(VelocityY, VelocityYPrev, DeltaTime);
	AddSources(VelocityZ, VelocityZPrev, DeltaTime);

	DiffuseGood(BT_XWall, VelocityXPrev, VelocityX, Viscosity, DeltaTime);
	DiffuseGood(BT_YWall, VelocityYPrev, VelocityY, Viscosity, DeltaTime);
	DiffuseGood(BT_ZWall, VelocityZPrev, VelocityZ, Viscosity, DeltaTime);

	Project(VelocityXPrev, VelocityYPrev, VelocityZPrev, VelocityX, VelocityY);
	
	Advect(BT_XWall, VelocityX, VelocityXPrev, VelocityXPrev, VelocityYPrev, VelocityZPrev, DeltaTime);
	Advect(BT_YWall, VelocityY, VelocityYPrev, VelocityXPrev, VelocityYPrev, VelocityZPrev, DeltaTime);
	Advect(BT_ZWall, VelocityZ, VelocityZPrev, VelocityXPrev, VelocityYPrev, VelocityZPrev, DeltaTime);

	Project(VelocityX, VelocityY, VelocityZ, VelocityXPrev, VelocityYPrev);
}

void AStamFluidBox::DensityStep(float DeltaTime)
{
	// Stam's original code uses pointer swapping - here we just swap the variable names instead.
	// Swapping is used instead of additional temporary arrays. Swapping means the result is always computed to a "new" array.
	AddSources(Density, DensityPrev, DeltaTime);
	
	// Try both variants with different parameters, see how the bad variant breaks quickly with higher values.
	if (bUseLinearSolver)
	{
		DiffuseGood(BoundaryType::BT_None, DensityPrev, Density, DiffusionFactor, DeltaTime);
	}
	else
	{
		DiffuseBad(BoundaryType::BT_None, DensityPrev, Density, DiffusionFactor, DeltaTime);
	}
	
	Advect(BoundaryType::BT_None, Density, DensityPrev, VelocityX, VelocityY, VelocityZ, DeltaTime);
}

void AStamFluidBox::DiffuseGood(BoundaryType BType, TArray<float>& X, TArray<float>& X0, float Diffusion,
	float DeltaTime)
{
	// The actual factor is the input one scaled by the time step and scaled to the size of our simulation grid.
	float Diff = DeltaTime * Diffusion * GridSize.X * GridSize.Y * GridSize.Z;
 
	for (int Step = 0; Step < LinearSolverSteps; Step++)
	{
		for (int i = 1; i <= GridSize.X; i++)
		{ 
			for (int j = 1; j <= GridSize.Y; j++)
			{
				for (int k = 1; k <= GridSize.Z; k++)
				{
					X[CellIndex(i,j,k)] = (X0[CellIndex(i,j,k)] + Diff*(
						X[CellIndex(i-1,j,k)]+X[CellIndex(i+1,j,k)]+
						X[CellIndex(i,j-1,k)]+X[CellIndex(i,j+1,k)]+
						X[CellIndex(i,j,k-1)]+X[CellIndex(i,j,k+1)]))/(1+6*Diff);
				}
			}
		}
		SetBounds(BType, X);
	}
}

void AStamFluidBox::DiffuseBad(BoundaryType BType, TArray<float>& X, TArray<float>& X0, float Diffusion,
	float DeltaTime)
{
	float Diff = DeltaTime * Diffusion * GridSize.X * GridSize.Y * GridSize.Z;
 
	for (int i = 1; i <= GridSize.X; i++)
	{ 
		for (int j = 1; j <= GridSize.Y; j++)
		{
			for (int k = 1; k <= GridSize.Z; k++)
			{
				// Naive forward approach. Faster than the stable solution and works for sufficiently small
				// time steps (and diffusion factors), but can easily become unstable (imagine a pause elsewhere in the
				// game, resulting in a very high time step).
				X[CellIndex(i, j, k)] = X0[CellIndex(i, j, k)]
				+ Diff*(X0[CellIndex(i + 1, j, k)]
				+ X0[CellIndex(i - 1, j, k)]
				+ X0[CellIndex(i, j + 1, k)]
				+ X0[CellIndex(i, j - 1, k)]
				+ X0[CellIndex(i, j, k + 1)]
				+ X0[CellIndex(i, j, k - 1)]
				- 6 * X0[CellIndex(i, j, k)]);
			}
		}
	}
	SetBounds(BType, X);
}

void AStamFluidBox::Advect(BoundaryType BType, TArray<float>& D, TArray<float>& D0, TArray<float>& VX,
	TArray<float>& VY, TArray<float>& VZ, float DeltaTime)
{
	float DeltaX = DeltaTime * GridSize.X;
	float DeltaY = DeltaTime * GridSize.Y;
	float DeltaZ = DeltaTime * GridSize.Z;
	
	for (int i = 1; i <= GridSize.X; i++)
	{
		for (int j = 1; j <= GridSize.Y; j++)
		{
			for (int k = 1; k <= GridSize.Z; k++)
			{
				// Find source position
				float x = i - DeltaX * VX[CellIndex(i, j, k)];
				float y = j - DeltaY * VY[CellIndex(i, j, k)];
				float z = k - DeltaZ * VZ[CellIndex(i, j, k)];

				// Make sure we're sufficiently inside the grid
				if (x < 0.5f) x = 0.5f; if (x > GridSize.X + 0.5f) x = GridSize.X + 0.5f;
				if (y < 0.5f) y = 0.5f; if (y > GridSize.Y + 0.5f) y = GridSize.Y + 0.5f;
				if (z < 0.5f) z = 0.5f; if (z > GridSize.Z + 0.5f) z = GridSize.Z + 0.5f;

				// Compute neighbouring cells and their contributions
				int i0 = static_cast<int>(x);
				int i1 = i0 + 1;
				
				int j0 = static_cast<int>(y);
				int j1 = j0 + 1;
				
				int k0 = static_cast<int>(z);
				int k1 = k0 + 1;
				
				// Compute contributing coefficients
				float s1 = x - i0;
				float s0 = 1 - s1;
				
				float t1 = y - j0;
				float t0 = 1 - t1;
				
				float u1 = z - k0;
				float u0 = 1 - u1;

				// Linear interpolation between the 8 neighbours
				D[CellIndex(i, j, k)] =
					s0 * (t0 * (u0 * D0[CellIndex(i0, j0, k0)] + u1 * D0[CellIndex(i0, j0, k1)]) +
						  t1 * (u0 * D0[CellIndex(i0, j1, k0)] + u1 * D0[CellIndex(i0, j1, k1)])) +
					s1 * (t0 * (u0 * D0[CellIndex(i1, j0, k0)] + u1 * D0[CellIndex(i1, j0, k1)]) +
						  t1 * (u0 * D0[CellIndex(i1, j1, k0)] + u1 * D0[CellIndex(i1, j1, k1)]));
			}
		}
	}
	SetBounds(BType, D);
}

void AStamFluidBox::Project(TArray<float>& Vx, TArray<float>& Vy, TArray<float>& Vz, TArray<float>& Temp,
	TArray<float>& Divisions)
{
	float InvSizeX = 1.0f / GridSize.X;
	float InvSizeY = 1.0f / GridSize.Y;
	float InvSizeZ = 1.0f / GridSize.Z;
	
	constexpr float Third = 1.0f / 3.0f;
	for (int i = 1; i <= GridSize.X; i++)
	{ 
		for (int j = 1; j <= GridSize.Y; j++)
		{
			for (int k = 1; k <= GridSize.Z; k++)
			{
				Divisions[CellIndex(i, j, k)] = -Third *
					(InvSizeX * (Vx[CellIndex(i + 1, j, k)] - Vx[CellIndex(i - 1, j, k)]) +
					InvSizeY * (Vy[CellIndex(i, j + 1, k)] - Vy[CellIndex(i, j - 1, k)]) +
					InvSizeZ * (Vz[CellIndex(i, j, k + 1)] - Vz[CellIndex(i, j, k - 1)]));
				Temp[CellIndex(i, j, k)] = 0;
			}
		} 
	}
	SetBounds(BoundaryType::BT_None, Divisions); SetBounds(BoundaryType::BT_None, Temp);

	// The gradient field is found using the Gauss-Seidel method again.
	for (int Step = 0; Step < LinearSolverSteps; Step++)
	{ 
		for (int i = 1; i <= GridSize.X; i++)
		{
			for (int j = 1; j <= GridSize.Y; j++)
			{
				for (int k = 1; k <= GridSize.Z; k++)
				{
					Temp[CellIndex(i, j, k)] = (Divisions[CellIndex(i, j, k)] +
						Temp[CellIndex(i - 1, j, k)] +
						Temp[CellIndex(i + 1, j, k)] +
						Temp[CellIndex(i, j + 1, k)] +
						Temp[CellIndex(i, j - 1, k)] +
						Temp[CellIndex(i, j, k + 1)] +
						Temp[CellIndex(i, j, k - 1)]) / 6;	
				} 
			}
		}
		SetBounds(BoundaryType::BT_None, Temp);
	} 
 
	for (int i = 1; i <= GridSize.X; i++)
	{
		for (int j = 1;j <= GridSize.Y; j++)
		{
			for (int k = 1; k <= GridSize.Z; k++)
			{
				Vx[CellIndex(i, j, k)] -= 0.5f*(Temp[CellIndex(i + 1, j, k)] - Temp[CellIndex(i - 1, j, k)])/InvSizeX;
				Vy[CellIndex(i, j, k)] -= 0.5f*(Temp[CellIndex(i, j + 1, k)] - Temp[CellIndex(i, j - 1, k)])/InvSizeY;
				Vz[CellIndex(i, j, k)] -= 0.5f*(Temp[CellIndex(i, j, k + 1)] - Temp[CellIndex(i, j, k - 1)])/InvSizeZ;
			}
		} 
	}
	SetBounds(BT_XWall, Vx); SetBounds(BT_YWall, Vy); SetBounds(BT_ZWall, Vz);
}

void AStamFluidBox::SetBounds(BoundaryType Type, TArray<float>& Data)
{
	const float XWall = Type == BoundaryType::BT_XWall ? -1 : 1;
	const float YWall = Type == BoundaryType::BT_YWall ? -1 : 1;
	const float ZWall = Type == BoundaryType::BT_ZWall ? -1 : 1;

	const int X = GridSize.X;
	const int Y = GridSize.Y;
	const int Z = GridSize.Z;
	
	// First take care of the walls of the fluid's bounding box
	// The type parameter value is used here to determine when to "reflect" the vector along the wall
	for (int i = 1; i <= Y; ++i)
	{
		for (int j = 1; j <= Z; ++j)
		{
			Data[CellIndex(  0, i, j)] = XWall * Data[CellIndex(1, i, j)];
			Data[CellIndex(X + 1, i, j)] = XWall * Data[CellIndex(  X, i, j)];
		}
	}

	for (int i = 1; i <= X; ++i)
	{
		for (int j = 1; j <= Z; ++j)
		{
			Data[CellIndex(i,   0, j)] = YWall * Data[CellIndex(i, 1, j)];
			Data[CellIndex(i, Y + 1, j)] = YWall * Data[CellIndex(i,   Y, j)];
		}

		for (int j = 1; j <= Y; ++j)
		{
			Data[CellIndex(i, j,   0)] = ZWall * Data[CellIndex(i, j, 1)];
			Data[CellIndex(i, j, Z + 1)] = ZWall * Data[CellIndex(i, j,   Z)];
		}
	}
	
	// Next set limiting values along the 6 edges of the box
	for (int i = 1; i <= X; ++i)
	{
		Data[CellIndex(i,   0,   0)] = 0.5f * (Data[CellIndex(i, 1,   0)] + Data[CellIndex(i,   0, 1)]);
		Data[CellIndex(i, Y + 1,   0)] = 0.5f * (Data[CellIndex(i,   Y,   0)] + Data[CellIndex(i, Y + 1, 1)]);
		Data[CellIndex(i,   0, Z + 1)] = 0.5f * (Data[CellIndex(i, 1, Z + 1)] + Data[CellIndex(i,   0,   Z)]);
		Data[CellIndex(i, Y + 1, Z + 1)] = 0.5f * (Data[CellIndex(i,   Y, Z + 1)] + Data[CellIndex(i, Y + 1,   Z)]);
	}
	
	for (int i = 1; i <= Y; ++i)
	{
		Data[CellIndex(0  , i, Z + 1)] = 0.5f * (Data[CellIndex(1, i, Z + 1)] + Data[CellIndex(  0, i,   Z)]);
		Data[CellIndex(X + 1, i, Z + 1)] = 0.5f * (Data[CellIndex(  X, i, Z + 1)] + Data[CellIndex(X + 1, i,   Z)]);
		Data[CellIndex(0  , i,   0)] = 0.5f * (Data[CellIndex(1, i,   0)] + Data[CellIndex(  0, i, 1)]);
		Data[CellIndex(X + 1, i,   0)] = 0.5f * (Data[CellIndex(  X, i,   0)] + Data[CellIndex(X + 1, i, 1)]);
	}

	for (int i = 1; i <= Z; ++i)
	{
		Data[CellIndex(  0, Y + 1, i)] = 0.5f * (Data[CellIndex(1, Y + 1, i)] + Data[CellIndex(  0,   Y, i)]);
		Data[CellIndex(X + 1, Y + 1, i)] = 0.5f * (Data[CellIndex(  X, Y + 1, i)] + Data[CellIndex(X + 1,   Y, i)]);
		Data[CellIndex(  0,   0, i)] = 0.5f * (Data[CellIndex(1,   0, i)] + Data[CellIndex(  0, 1, i)]);
		Data[CellIndex(X + 1,   0, i)] = 0.5f * (Data[CellIndex(  X,   0, i)] + Data[CellIndex(X + 1, 1, i)]);
	}

	// Finally handle the 8 corners of the box
	constexpr float Third = 1.0 / 3.0;
	Data[CellIndex(  0,   0,   0)] = Third * (Data[CellIndex(1,   0,   0)] + Data[CellIndex(    0, 1,   0)] + Data[CellIndex(    0,   0, 1)]);
	Data[CellIndex(  0, Y + 1,   0)] = Third * (Data[CellIndex(1, Y + 1,   0)] + Data[CellIndex(    0,   Y,   0)] + Data[CellIndex(    0, Y + 1, 1)]);
	Data[CellIndex(X + 1,   0,   0)] = Third * (Data[CellIndex(  X,   0,   0)] + Data[CellIndex(  X + 1, 1,   0)] + Data[CellIndex(  X + 1,   0, 1)]);
	Data[CellIndex(X + 1, Y + 1,   0)] = Third * (Data[CellIndex(  X, Y + 1,   0)] + Data[CellIndex(  X + 1,   Y,   0)] + Data[CellIndex(  X + 1, Y + 1, 1)]);
	Data[CellIndex(  0,   0, Z + 1)] = Third * (Data[CellIndex(1,   0, Z + 1)] + Data[CellIndex(    0, 1, Z + 1)] + Data[CellIndex(    0,   0,   Z)]);
	Data[CellIndex(  0, Y + 1, Z + 1)] = Third * (Data[CellIndex(1, Y + 1, Z + 1)] + Data[CellIndex(    0,   Y, Z + 1)] + Data[CellIndex(    0, Y + 1,   Z)]);
	Data[CellIndex(X + 1,   0, Z + 1)] = Third * (Data[CellIndex(  X,   0, Z + 1)] + Data[CellIndex(  X + 1, 1, Z + 1)] + Data[CellIndex(  X + 1,   0,   Z)]);
	Data[CellIndex(X + 1, Y + 1, Z + 1)] = Third * (Data[CellIndex(  X, Y + 1, Z + 1)] + Data[CellIndex(  X + 1,   Y, Z + 1)] + Data[CellIndex(  X + 1, Y + 1,   Z)]);
}

void AStamFluidBox::AddSources(TArray<float>& Cells, TArray<float>& Sources, float DeltaTime)
{
	const int N = DensityGridSize.X * DensityGridSize.Y * DensityGridSize.Z;
	for (int i = 0; i < N; i++)
	{
		Cells[i] += Sources[i] * DeltaTime;
	}
}

int AStamFluidBox::CellIndex(int X, int Y, int Z) const
{
	return X + DensityGridSize.X * Y + DensityGridSize.Y * DensityGridSize.X * Z;
}

int AStamFluidBox::CellIndex(FIntVector Pos) const
{
	return CellIndex(Pos.X, Pos.Y, Pos.Z);
}

