#include "NewtonParticleSystem.h"

// Sets default values
ANewtonParticleSystem::ANewtonParticleSystem()
{
 	// Set this actor to call Tick() every frame.  You can turn this off to improve performance if you don't need it.
	PrimaryActorTick.bCanEverTick = true;

	// Create an empty SceneComponent to allow for placement in the scene.
	RootComponent = CreateDefaultSubobject<USceneComponent>(TEXT("Root"));
	
	ParticleClass = ConstructorHelpers::FClassFinder<AStaticMeshActor>(TEXT("/Game/Blueprints/BP_Particle")).Class;
}

// Called when the game starts or when spawned
void ANewtonParticleSystem::BeginPlay()
{
	Super::BeginPlay();
	
	// Generate particles, place them randomly around this actor and remember them in an array.
	for (int i = 0; i < ParticleCount; ++i)
	{
		auto Location = FMath::VRand() * InitialSpread * FMath::FRand();
		auto Particle = GetWorld()->SpawnActor<AStaticMeshActor>(ParticleClass, GetActorLocation() + Location, { });
		Particle->GetStaticMeshComponent()->SetMassOverrideInKg(NAME_None, ParticleMass, true);
		Particles.Add(Particle);
	}
}

FVector ANewtonParticleSystem::ComputeParticleForce(uint32 ParticleIndex) const
{
	// Start with no force, compute contribution of each particle.
	FVector TotalForce = FVector::Zero();
	for (int i = 0; i < Particles.Num(); ++i)
	{
		if (i == ParticleIndex)
			continue;

		// First we need the vector pointing to the foreign particle,
		// so we can use its direction and length.
		auto Direction = Particles[i]->GetActorLocation() - Particles[ParticleIndex]->GetActorLocation();

		// Force = M_1 * M_2 / r^2
		auto Force = Direction.GetSafeNormal()
		* Particles[i]->GetStaticMeshComponent()->GetMass()
		* Particles[ParticleIndex]->GetStaticMeshComponent()->GetMass()
		/ Direction.SquaredLength()
		* GravitationalConstant;
		TotalForce += Force;
	}

	return TotalForce;
}

// Called every frame
void ANewtonParticleSystem::Tick(float DeltaTime)
{
	Super::Tick(DeltaTime);


	// Filter invalid particles (destroyed)
	for (int i = Particles.Num() - 1; i >= 0; --i)
	{
		if (!IsValid(Particles[i]))
		{
			Particles.RemoveAt(i);
		}
	}

	// Here are two ways of going about the computation. Particles can be processed either
	// sequentially or in parallel. ParallelFor is a UE construct which tells the runtime
	// to schedule a set amount of tasks and set a barrier at the end. Task will be eventually completed
	// by worker threads and execution in this function will continue once all the tasks are done.
	//
	// Since each computation depends only on the previous simulation step and not on any data in this step,
	// computing it in parallel is simple.
	if (UseParallelFor)
	{
		ParallelFor(Particles.Num(), [this](int32 i) -> void const
		{
			auto ForceToApply = ComputeParticleForce(i);
			Particles[i]->GetStaticMeshComponent()->AddForce(ForceToApply);
		});	
	}
	else
	{
		for(int i = 0; i < Particles.Num(); ++i)
		{
			auto ForceToApply = ComputeParticleForce(i);
			Particles[i]->GetStaticMeshComponent()->AddForce(ForceToApply);
		}	
	}
}

