#include "PressurePlate.h"

#include "Kismet/GameplayStatics.h"

DEFINE_LOG_CATEGORY(PressurePlate)

APressurePlate::APressurePlate()
{
	PrimaryActorTick.bCanEverTick = true;

	// Find a mesh that we can use for display - in this case just a cube.
	// This is an engine mesh, so we can be sure it will always be available. However, a more correct course
	// of action would be to check if the object was found using a call to Succeeded() on the finder.
	auto Mesh = ConstructorHelpers::FObjectFinder<UStaticMesh>(TEXT("/Script/Engine.StaticMesh'/Engine/EngineMeshes/Cube.Cube'")).Object;

	// Finally, set the mesh for display in our static mesh component.
	GetStaticMeshComponent()->SetStaticMesh(Mesh);

	// And perhaps scale it a little, so the cube is more of a flat box.
	GetStaticMeshComponent()->SetRelativeScale3D(FVector(1.f, 1.f, .2f));
}

void APressurePlate::BeginPlay()
{
	// We cannot add event callbacks in the constructor as the engine doesn't guarantee it will be called when
	// an instance is inserted into the scene. Instead, we get events automatically routed to AActor's methods
	// for various setup tasks. BeginPlay is one of them - called when the game starts, or when an instance is
	// created in the scene. Here we can do setup that requires interaction with other gameplay systems.

	// AddDynamic is a macro that, loosely, serves a similar function as the std::bind construct, for those
	// familiar with it.
	GetStaticMeshComponent()->OnComponentHit.AddDynamic(this, &APressurePlate::OnPressed);
	
	Super::BeginPlay();
}

void APressurePlate::Tick(float DeltaSeconds)
{
	Super::Tick(DeltaSeconds);
}

void APressurePlate::OnPressed(UPrimitiveComponent* HitComponent, AActor* OtherActor, UPrimitiveComponent* OtherComp,
	FVector NormalImpulse, const FHitResult& Hit)
{
	// We can log either to the output log...
	UE_LOG(PressurePlate, Warning, TEXT("I was pressed by: %s"), *OtherActor->GetName());
	// or to the screen for a temporary on-screen text message.
	GEngine->AddOnScreenDebugMessage(-1, 5.f, FColor::Red, TEXT("I was pressed!"));

	// If we have a class set AND we haven't spawned anything yet (indicated by bSupressPressed),
	// then we spawn a boulder above.
	if (!bSuppressPressed && BoulderClass.Get() != nullptr)
	{
		auto SpawnLocation = GetActorLocation() + FVector{0, 0, 1000};
		GetWorld()->SpawnActor(BoulderClass, &SpawnLocation);
		bSuppressPressed = true;
	}
}
