from dataclasses import dataclass
from datetime import datetime
from pathlib import Path
from typing import Generator, Self

FOOD_FNAME = Path("food.csv")
ACTIVITIES_FNAME = Path("activities.csv")


@dataclass
class Entry:
    name: str
    calories: int
    date: str = ""

    def __post_init__(self):
        if not self.date:
            self.date = today()

    def to_csv(self):
        return f"{self.date},{self.name},{self.calories}"

    @classmethod
    def from_csv(cls, line: str) -> Self:
        date, name, calories = line.strip().split(",")
        return cls(name, int(calories), date)


def today() -> str:
    return datetime.now().strftime("%Y-%m-%d")


def add_entry(fpath: Path, entry: Entry) -> None:
    with fpath.open("a", encoding="utf-8") as f:
        f.write(f"{entry.to_csv()}\n")


def log_food(name: str, calories: int, date: str = "") -> None:
    entry = Entry(name, calories, date)
    add_entry(FOOD_FNAME, entry)
    print(f"Appended food: {entry.name} ({entry.calories} kcal) on {entry.date}")


def log_activity(name: str, calories: int, date: str = "") -> None:
    entry = Entry(name, calories, date)
    add_entry(ACTIVITIES_FNAME, entry)
    print(f"Appended activity: {entry.name} ({entry.calories} kcal) on {entry.date}")


def run_day_summary(date: str):
    report_data = calculate_report_data(date)
    print_report(*report_data)


def calculate_report_data(date: str) -> tuple[str, int, int, int]:
    food = load_entries(FOOD_FNAME)
    activity = load_entries(ACTIVITIES_FNAME)
    food_total = sum(entry.calories for entry in food if entry.date == date)
    activity_total = sum(entry.calories for entry in activity if entry.date == date)
    net = food_total - activity_total
    return date, food_total, activity_total, net


def load_entries(fpath: Path) -> Generator[Entry]:
    try:
        with fpath.open("r", encoding="utf-8") as f:
            for line in f:
                yield Entry.from_csv(line)
    except FileNotFoundError:
        print(f"Could not read {fpath}")


def print_report(date: str, food_total: int, activity_total: int, net: int) -> None:
    print(f"\nSummary for {date}")
    print(f"  🍎 Food:     {food_total} kcal")
    print(f"  🏃 Activity: {activity_total} kcal")
    print(f"  ⚖️  Net:       {net} kcal")


def main_1():
    log_food("Banana", 100)
    log_activity("Running", 300)
    run_day_summary(today())


if __name__ == "__main__":
    main_1()
