
import json
import networkx as nx

# ==========================================================
# Task 1 – Multilayer Test Model
# ==========================================================

class HierarchicalMultilayerTestModel:
    def __init__(self, layers):
        """
        layers: ordered list from lowest to highest layer
        """
        self.layers = layers
        self.intralayer = None #TODO: dictionary (layer name -> nx.Graph)
        self.interlayer = {}  # (upper, lower) -> set of (u, v)

    def add_component(self, layer, component):
        #TODO: insert component to the layer
        pass

    def add_intralayer_relation(self, layer, u, v):
        #TODO: add intralayer relation  (u and v are components in layer)
        pass

    def add_interlayer_projection(self, upper, lower, u, v):
        """
        upper, lower are layers
        u is in upper layer, v is in lower layer
        """
        #TODO: add interlayer projection between upper (component u) and lower (component l) layers
        pass


# ==========================================================
# Task 2 + 3 – Test Requirements & Diagnostics
# ==========================================================

def check_R1_component_coverage(model):
    """
    R1: Each component participates in at least one intralayer relation in its own layer.
    """
    #TODO: implement
    return None


def check_R2_intralayer_coverage(model):
    """
    R2: Each layer contains at least one intralayer relation.
    """
    #TODO: implement
    return None


def check_R3_cross_layer_traceability(model):
    """
    R3: Each component in a higher layer has at least one projection to the layer directly below.
    """
    #TODO: implement
    return None


def run_diagnostics(model):
    return {
        "R1_component_coverage": check_R1_component_coverage(model),
        "R2_intralayer_coverage": check_R2_intralayer_coverage(model),
        "R3_cross_layer_traceability": check_R3_cross_layer_traceability(model),
    }


# ==========================================================
# Task 4 – Test Path Generation
# ==========================================================

def generate_end_to_end_paths(model, start_layer):
    """
    Finds all paths from all components in start_layer
    to all components in the lowest layer.

    Output format:
    {
        "(start_layer:start -> lowest_layer:goal)": [
            [ "layer:component", ... ],
            ...
        ]
    }
    """

    #TODO: implement whole function

    # Raw list of paths
    raw_paths = []

    # --------------------------------------------------
    # Group paths by (start, goal)
    # --------------------------------------------------
    result = {}

    def fmt(node):
        layer, comp = node
        return f"{layer}:{comp}"

    for path in raw_paths:
        start = path[0]
        goal = path[-1]
        key = f"({fmt(start)} -> {fmt(goal)})"

        result.setdefault(key, []).append([fmt(n) for n in path])

    return result



# ==========================================================
# Task 5 – Example + Fault Injection
# ==========================================================

if __name__ == "__main__":
    layers = ["physical", "logical", "service"]
    model = HierarchicalMultilayerTestModel(layers)

    # Physical
    model.add_component("physical", "Server")
    model.add_component("physical", "Switch")
    model.add_intralayer_relation("physical", "Server", "Switch")

    # Logical
    model.add_component("logical", "Routing")
    model.add_component("logical", "Firewall")
    model.add_intralayer_relation("logical", "Routing", "Firewall")

    # Service
    model.add_component("service", "WebService")
    model.add_component("service", "API")
    model.add_intralayer_relation("service", "WebService", "API")

    # Interlayer projections
    model.add_interlayer_projection("logical", "physical", "Routing", "Server")
    model.add_interlayer_projection("logical", "physical", "Firewall", "Switch")
    model.add_interlayer_projection("service", "logical", "API", "Routing")
    model.add_interlayer_projection("service", "logical", "WebService", "Firewall")

    print("Diagnostics (before fault):")
    print(run_diagnostics(model))

    print("\nEnd-to-end test paths:")
    print(print(json.dumps(generate_end_to_end_paths(model, "service"), indent=2)))

    # Fault injection: remove projection
    model.interlayer[("service", "logical")].clear()

    print("\nDiagnostics (after fault):")
    print(run_diagnostics(model))



    """
    EXPECTED OUTPUT TO THE GIVEN INPUT SETUP

    Diagnostics (before fault):
    {'R1_component_coverage': True, 'R2_intralayer_coverage': True, 'R3_cross_layer_traceability': True}

    End-to-end test paths:
    {
        "(service:WebService -> physical:Switch)": [
            [
            "service:WebService",
            "service:API",
            "logical:Routing",
            "logical:Firewall",
            "physical:Switch"
            ],
            [
            "service:WebService",
            "logical:Firewall",
            "physical:Switch"
            ]
        ],
        "(service:WebService -> physical:Server)": [
            [
            "service:WebService",
            "service:API",
            "logical:Routing",
            "physical:Server"
            ],
            [
            "service:WebService",
            "logical:Firewall",
            "logical:Routing",
            "physical:Server"
            ]
        ],
        "(service:API -> physical:Server)": [
            [
            "service:API",
            "service:WebService",
            "logical:Firewall",
            "logical:Routing",
            "physical:Server"
            ],
            [
            "service:API",
            "logical:Routing",
            "physical:Server"
            ]
        ],
        "(service:API -> physical:Switch)": [
            [
            "service:API",
            "service:WebService",
            "logical:Firewall",
            "physical:Switch"
            ],
            [
            "service:API",
            "logical:Routing",
            "logical:Firewall",
            "physical:Switch"
            ]
        ]
    }

    Diagnostics (after fault):
    {'R1_component_coverage': True, 'R2_intralayer_coverage': True, 'R3_cross_layer_traceability': False}

    """
