#include "oscilloscope.h"
#include "ui_oscilloscope.h"

#include <QtCharts/QAreaSeries>
#include <QtCharts/QChartView>
#include <QtCharts/QLineSeries>
#include <QtCharts/QDateTimeAxis>

#include <QtSerialPort/QSerialPort>

#include <QMessageBox>

#include <chrono>

static constexpr std::chrono::seconds kWriteTimeout = std::chrono::seconds{1};

oscilloscope::oscilloscope(QWidget *parent)
    : QMainWindow(parent)
    , ui(new Ui::oscilloscope)
    , m_timer(new QTimer(this))
    , m_serial(new QSerialPort(this))
    , m_settings(new SettingsDialog(this))
{
    ui->setupUi(this);
    ui->valSampling->setValue(10);
    ui->valSignal->setValue(1);
    connect(m_serial, &QSerialPort::readyRead, this, &oscilloscope::readData);
    connect(m_timer, &QTimer::timeout, this, &oscilloscope::handleWriteTimeout);
    m_timer->setSingleShot(true);

    connect(m_serial, &QSerialPort::bytesWritten, this, &oscilloscope::handleBytesWritten);

    this->dataActual = new QLineSeries();
    dataActual->setName("Napeti");

    dataValues = new QAreaSeries(dataActual);
    QPen pen(0x059605);
    pen.setWidth(3);
    dataValues->setPen(pen);

    chart = new QChart();
    chart->addSeries(dataActual);
    chart->setTitle(QString("Namerene napeti"));
    chart->createDefaultAxes();

    chart->axisY()->setRange(-4200, 4200);
    ui->chartOscilloscope->setChart(chart);

}

oscilloscope::~oscilloscope()
{
    delete m_settings;
    delete ui;
}

void oscilloscope::on_btnStart_clicked()
{
    this->writeData(QString("start\n").toLatin1());
}


void oscilloscope::on_btnStop_clicked()
{
    this->writeData(QString("stop\n").toLatin1());
}

void oscilloscope::openSerialPort()
{
    const SettingsDialog::Settings p = m_settings->settings();
    m_serial->setPortName(p.name);
    m_serial->setBaudRate(p.baudRate);
    m_serial->setDataBits(p.dataBits);
    m_serial->setParity(p.parity);
    m_serial->setStopBits(p.stopBits);
    m_serial->setFlowControl(p.flowControl);
    if (m_serial->open(QIODevice::ReadWrite)) {
        ui->btnStart->setEnabled(true);
        ui->btnStop->setEnabled(true);
    } else {
        QMessageBox::critical(this, tr("Error"), m_serial->errorString());
    }
}

void oscilloscope::closeSerialPort()
{
    if (m_serial->isOpen())
        m_serial->close();
}

void oscilloscope::readData()
{
    static double xMin = 0, xMax;
    const QByteArray data = m_serial->readAll();
    dataBuffer.append(data);
    if(dataBuffer.contains('\n')){
        QString newData = "";
        QStringList separated = dataBuffer.split('\n');
        QStringList::iterator it = separated.begin();

        for(;it != separated.end(); ++it){
            QString strPoint = *it;
            if(strPoint.endsWith(';')){
                QStringList point = strPoint.split(',');

                if(point.length() == 2){
                    double time = point[0].toDouble();
                    point[1].removeLast();
                    double value = point[1].toDouble();

                    xMax = time;
                    chart->axisX()->setRange(xMin, xMax);
                    *dataActual << QPointF(time, value);
                }
            }
            else{
                newData = *it;   // last value
            }
        }
        dataBuffer = newData;

    }
}

void oscilloscope::writeData(const QByteArray &data)
{
    const qint64 written = m_serial->write(data);
    if (written == data.size()) {
        m_bytesToWrite += written;
        m_timer->start(kWriteTimeout);
    } else {
        const QString error = tr("Failed to write all data to port %1.\n"
                                 "Error: %2").arg(m_serial->portName(),
                                       m_serial->errorString());
        qDebug() << error;
    }
}

void oscilloscope::handleBytesWritten(qint64 bytes)
{
    m_bytesToWrite -= bytes;
    if (m_bytesToWrite == 0)
        m_timer->stop();
}

void oscilloscope::handleWriteTimeout()
{
    const QString error = tr("Write operation timed out for port %1.\n"
                             "Error: %2").arg(m_serial->portName(),
                                   m_serial->errorString());
    qDebug() << error;
}


void oscilloscope::on_btnOpenPort_clicked()
{
    openSerialPort();

    QString command = QString("stop\n");
    this->writeData(command.toLatin1());
}


void oscilloscope::on_btnSettings_clicked()
{
    m_settings->show();
}

