# Authors: Jiri Kubik (kubikji2@fel.cvut.cz)

from abc import ABC, abstractmethod

import numpy as np

class InchwormInterface(ABC):
    
    @abstractmethod
    # returns
    # - np.array of [x,y,z] world coordinates in meters if `part_name` is a valid part name,
    # - None otherwise
    def get_part_position(self, part_name : str) -> "np.array | None":
        pass
    
    @abstractmethod
    # returns
    # - np.array of intrinsic XYZ euler angles (see [1]) if `part_name` is a valid part name,
    # - None otherwise
    # Note: 
    # [1] https://en.wikipedia.org/wiki/Euler_angles#Conventions_by_intrinsic_rotations
    def get_part_rotation(self, part_name : str, degrees : bool = True) -> "np.array | None":
        pass
    
    @abstractmethod
    # returns
    # - 6D np.array of angular x,y,z velocities in [radians/second]
    #   followed by linear x,y,z velocities [meters/second] if `part_name` is a valid part name,
    # - None otherwise
    def get_part_velocity(self, part_name : str) -> "np.array | None":
        pass
    
    @abstractmethod
    # returns joint position [degrees] if `joint_name` is a valid part name,
    # None otherwise
    def get_joint_position(self, joint_name : str, degrees : bool = True) -> "float | None":
        pass
    
    @abstractmethod
    # returns joint velocity [radians/second] if `joint_name` is a valid part name,
    # None otherwise
    def get_joint_velocity(self, joint_name : str) -> "float | None":
        pass
    
    @abstractmethod
    # returns:
    # - True iff the parts of given name are touching, 
    # - False iff parts are not touching, and
    # - None iff at least one of the part_name is not a valid part name
    def is_touching(self, part_name_1 : str, part_name_2 : str) -> "bool | None":
        pass
