
// Author: Michal Cvach
// cvachmic@fel.cvut.cz

/* A simple script that can preprocess a .json file containing lines with complicated geometry.
 * Lines are simplified to just the begin point and end point, which means we only preserve
 * the information about where the route in our case begins and where it ends.
 * Additionally, we generate names for routes that don't have any, because
 * we later use those names as identifiers for cycling routes. */

#include <cstdio>
#include <vector>
#include <string>

using namespace std;

// This function generates "N" instead of null values for the one way routes specification.
//______________________________________________________________________________________________________________________
void generateBothWaysRouteFlag() {
    int c;
    char reference[] = "EDNOSMERKA\":";
    unsigned int curPos = 0;
    while (curPos < 12) {
        c = getchar();
        putchar(c);
        if (c != reference[curPos++]) {
            return;
        }
    }

    c = getchar();
    if (c != 'n') {
        putchar(c);
        return;
    }
    getchar();
    getchar();
    getchar();
    putchar('"');
    putchar('N');
    putchar('"');
}

// This function will generate a name for routes which don't have a valid name by default.
// If a route already has a name, this function won't change it.
//______________________________________________________________________________________________________________________
void generateRouteNameIfNull(unsigned int & cnt) {
    int c;
    char reference[] = "ISLO_TRAS\":";
    unsigned int curPos = 0;
    while (curPos < 11) {
        c = getchar();
        putchar(c);
        if (c != reference[curPos++]) {
            return;
        }
    }

    c = getchar();
    if (c != 'n') {
        putchar(c);
        return;
    }
    getchar();
    getchar();
    getchar();
    putchar('"');
    putchar('U');
    putchar('N');
    string num = to_string(++cnt);
    for (unsigned int i = 0; i < num.size(); i++) {
        putchar(num[i]);
    }
    putchar('"');

}

// Loads exactly one geometry point and returns true if it was the last one in the line or false otherwise.
//______________________________________________________________________________________________________________________
bool loadPoint(vector<int> & point) {
    int c;
    c = getchar();
    while( c != ']') {
        point.push_back(c);
        c = getchar();
    }
    point.push_back(c);
    c = getchar();
    if (c == ',') {
        return false;
    }
    return true;
}

// Prints exactly one geometry point to output.
//______________________________________________________________________________________________________________________
void printPoint(vector<int> & point) {
    for(unsigned int i = 0; i < point.size(); i++) {
        putchar(point[i]);
    }
}

// Used to process one line, we first load and then print the first point, then keep loading points, until we load the
// last one, and then we only print the last one.
//______________________________________________________________________________________________________________________
void processOneRoute() {
    int c;
    c = getchar();
    putchar(c);
    vector<int> point;
    loadPoint(point);
    printPoint(point);
    putchar(',');
    bool last = false;
    while (! last) {
        point.clear();
        last = loadPoint(point);
    }
    printPoint(point);
    putchar(']');
}

// Validates, whether we have already encountered "coordinates", if so, we should start simplifying a line.
//______________________________________________________________________________________________________________________
bool validateCoordinates() {
    char reference[] = "oordinates\":";

    int c;
    int pos = 0;
    while (pos < 12) {
        c = getchar();
        putchar(c);
        if (c != reference[pos++]) {
            return false;
        }
    }
    return true;
}

// Here we just start read the input character by character, if we encounter 'c', we validate if in fact we have
// encountered the 'c' at the beginning of "coordinates" or not, and if yes, we must process (simplify) one line.
//______________________________________________________________________________________________________________________
int main() {
    int c;
    unsigned int unnamedRouteCnt = 0;

    while (true) {
        c = getchar();
        if (c == EOF) {
            break;
        }
        putchar(c);
        if ((char)c == 'c') {
            if (validateCoordinates()) {
                processOneRoute();
            }
        }
        if ((char)c == 'C') {
            generateRouteNameIfNull(unnamedRouteCnt);
        }
        if ((char)c == 'J') {
            generateBothWaysRouteFlag();
        }
    }

    return 0;
}
