#include <stdio.h>
#include <stdlib.h>
#include <string.h>


/** 
 * @brief Structure to store a contact's details.
 */
typedef struct{
	char name[100];
	int age;
	int date[3];  // [YYYY, MM, DD]
} contact;


/**
 * @brief Enumeration to index the date fields (Year, Month, Day).
 */
typedef enum {
    YEAR = 0,
    MONTH,
    DAY
} date_field;


/**
 * @brief Enumeration of months.
 */
typedef enum {
	JANUARY = 1,
	FEBRUARY,
	MARCH,
	APRIL,
	MAY,
	JUNE,
	JULY,
	AUGUST,
	SEPTEMBER,
	OCTOBER,
	NOVEMBER,
	DECEMBER
} months;


/**
 * @brief Clears the input buffer in case of invalid input.
 * 
 * It continues to read characters until a newline or EOF is encountered.
 */
void clear_input(){
	int ch;
	while ((ch = getchar()) != '\n' && ch != EOF);
}


/**
 * @brief Adds a new contact to the list.
 * 
 * Reallocates memory to store a new contact and reads user input.
 * 
 * @param contacts Pointer to the array of contact pointers.
 * @param num_contacts Pointer to the number of contacts.
 * 
 * @return 0 if successful, 1 on input error, or -1 on memory allocation error.
 */
int add_contact(contact** contacts, int* num_contacts){
	contact new_contact;
	
	// read user input - name
	printf("Enter name\n");
	clear_input();
	if (fgets(new_contact.name, sizeof(new_contact.name), stdin) == NULL) {
		printf("Input error\n");
		return 1;
	}
	new_contact.name[strcspn(new_contact.name, "\n")] = '\0';  // remove new line (if any)
	
	// read user input - age
	printf("Enter age\n");
	if (scanf("%d", &new_contact.age) != 1){
		printf("Input error\n");
		clear_input();
		return 1;
	}
	
	// read user input - date of birth
	printf("Enter date of birth in format YYYY-MM-DD\n");
	if (scanf("%4d-%2d-%2d", &new_contact.date[YEAR], &new_contact.date[MONTH], &new_contact.date[DAY]) != 3){
		printf("Input error\n");
		clear_input();
		return 1;
	}
	
	// validate date of birth
	if (new_contact.date[MONTH] < JANUARY || new_contact.date[MONTH] > DECEMBER || new_contact.date[DAY] < 0 || new_contact.date[DAY] > 31){
		printf("Invalid date\n");
		return 1;
	}

	// realloc the array
	(*contacts) = (contact*) realloc(*contacts, (*num_contacts + 1) * sizeof(contact));
	if (*contacts == NULL){
		printf("Allocation error\n");
		return -1;
	}
	
	// add new contact
	(*contacts)[*num_contacts] = new_contact;
	(*num_contacts)++;
	
	return 0;
}


/**
 * @brief Prints all the contacts stored in the contacts array.
 * 
 * @param contacts Pointer to the array of contacts.
 * @param num_contacts Number of contacts in the array.
 */
void print_contacts(contact* contacts, int num_contacts){
	printf("----------------------------------------\n");
	for (int i = 0; i < num_contacts; i++) {
		printf("===User %d===\n", i);
		printf("Name: %s\n", contacts[i].name);
		printf("Age: %d\n", contacts[i].age);
		printf("Date of Birth: %d-%d-%d\n", contacts[i].date[YEAR], contacts[i].date[MONTH], contacts[i].date[DAY]);
		printf("\n");
	}
	printf("----------------------------------------\n");
}


/**
 * @brief Deletes a contact from the list by index.
 * 
 * Reallocates memory to shrink the contacts array.
 * 
 * @param contacts Pointer to the array of contact pointers.
 * @param num_contacts Pointer to the number of contacts.
 * 
 * @return 0 if successful, 1 on input error, and -1 on memory allocation error.
 */
int delete_contact(contact** contacts, int* num_contacts){
	
	// read ID to delete
	printf("Enter contact ID to delete\n");
	int index;
	if (scanf("%d", &index) != 1 || index < 0 || index >= *num_contacts){
		printf("Invalid index\n");
		clear_input();
		return 1;
	}
	
	// shift following contacts and decrement number
	for (int i = index; i < (*num_contacts)-1; i++){
		(*contacts)[i] = (*contacts)[i+1];
	}
	(*num_contacts)--;
	
	// realloc array
	(*contacts) = (contact*) realloc(*contacts, *num_contacts * sizeof(contact));
	if (*num_contacts > 0 && *contacts == NULL){
		printf("Allocation error\n");
		return -1;
	}

	return 0;

}


int main(int argc, char* argv[]){

	// init variables
	contact* contacts = NULL;
	int num_contacts = 0;
	int key;
	int exit = 0;
	
	while(!exit){
		
		// print menu
		printf("1. add contact\n");
		printf("2. view all contacts\n");
		printf("3. delete a contact\n");
		printf("4. exit\n");
		printf("*Press key to continue*\n");
		
		// read key
		if (scanf("%d", &key) != 1){
			printf("Invalid key\n");
			clear_input();
			continue;
		}
		
		switch (key){
			// add contact
			case 1:
				if(add_contact(&contacts, &num_contacts) < 0){
					free(contacts);
					return -1;
				}
				break;
			
			// view all contacts
			case 2:
				print_contacts(contacts, num_contacts);
				break;
				
			// delete a contact
			case 3:
				if(delete_contact(&contacts, &num_contacts) < 0){
					free(contacts);
					return -1;
				}
				break;
			
			// exit
			case 4:
				printf("Exiting program\n");
				exit = 1;
				break;
			default:
				printf("Invalid key, try again\n");
		}
	
	}
	
	// free allocated memory
	free(contacts);

	return 0;
}
