<?php
/**
 * @author Martin Klima
 * Vyrobeno pro demonstraci práce s obrázky v předmětu ZWA.
 */

/**
 * Třída reprezentuje seznam obrázku. Umí je uložit do souboru i načíst ze souboru.
 */
class ImageList
{
    /** @var array $images Pole obrázků typu Image */
    protected array $images = array();
    /** @var string $filename Název souboru, do a ze kterého se seznam bude ukládat/načítat. */
    protected string $filename;

    //protected string $original_directory = self::ORIGINAL_DIRECTORY;
    //protected string $thumbnail_directory = self::THUMBNAIL_DIRECTORY;

    /**
     * Konstruktor, pokud je druhý parametr true, nahrají se obrázky okamžitě při vzniku objektu.
     * @param string $filename Název souboru, kde jsou data.
     * @param bool $load Zda nahrát okamžitě data. Dedault je false, tedy ne.
     */
    public function __construct(string $filename, bool $load = false)
    {
        $this->setFilename($filename);
        //$this->setOriginalDirectory($originalDirectory);
        //$this->setThumbnailDirectory($thumbnailDirectory);
        if ($load) {
            $this->load();
        }
    }

    /**
     * Nahraje obrázky ze souboru.
     * @return array Vrací pole obrázků typu Image
     */
    public function load(): array
    {
        @$file = fopen($this->filename, "r");
        if ($file) {
            $this->images = unserialize(fread($file, filesize($this->filename)));
            fclose($file);
            return $this->images;
        }
        $this->images = array();
        return $this->images;
    }

    /**
     * Uloží obrázky do souboru.
     * @return void
     */
    public function save(): void
    {
        $file = fopen($this->getFileName(), "w");
        $to_write = serialize($this->images);
        if ($file) {
            fwrite($file, $to_write);
            fclose($file);
        }
    }

    /**
     * Smaže obrázek na pozici $id ze seznamu obrázků. Pokud je druhý parametr true, okamžitě výsledek uloží.
     * Pokud $id neexistuje, nestane se nic.
     * @param int $id Id pozice obrázku.
     * @param $saveImmediately Pokud je true, bude výsledek okamžitě uložen.
     * @return bool Pokud došlo ke smazání, vrací true, jinak false.
     */
    public function deleteImage(int $id, $saveImmediately = true): bool
    {
        if (!isset($this->images[$id])) {
            return false;
        }
        $image = $this->images[$id];
        $image->destroy();
        unset($this->images[$id]);
        if ($saveImmediately) {
            $this->save();
        }
        return true;
    }

    /**
     * Přidá nový obrázek do pole obrázků.
     * @param Image $image Nový obrázek, který se přidá na konec pole obrázků
     * @param bool $saveImmediately Pokud je true, výsledek se okamžitě uloží.
     * @return Image Vrací obrázek, který byl zadán jako parametr.
     */
    public function add(Image $image, bool $saveImmediately = true): Image
    {
        $this->images[] = $image;
        if ($saveImmediately) {
            $this->save();
        }
        return $image;
    }

    /**
     * Vrací pole obrázků typu Image
     * @return array
     */
    public function getImages(): array
    {
        return $this->images;
    }


    /**
     * Nastaví název a cestu souboru, do a ze kterého se mají ukládat/načítat data.
     * @param string $file Název souboru.
     * @return string
     */
    public function setFileName(string $file): string
    {
        $this->filename = $file;
        return $file;
    }

    /**
     * Vrací název datového souboru.
     * @return string
     */
    public function getFileName(): string
    {
        return $this->filename;
    }


}