<?php

/**
 * @author Martin Klima
 * Vyrobeno pro demonstraci práce s obrázky v předmětu ZWA.
 */

/**
 * class Image
 * Popisuje obrázek jako takový.
 * Umí z nového obrázku vyrobit jeho miniaturu. Pamatuje si také uložení miniatury a původního obrázku.
 * Pro potřeby uložení vygeneruje unikátní název (id) pro každý obrázek, proto při ukládání
 * nedojde ke kolizi.
 * Obrázek umí i smazat, tj. jeho originál i miniaturu.
 */
class Image {
    /** @var string $name Jméno souboru. */
    protected string $name;
    /** @var string $original_directory Cesta pro uložení originálu. */
    protected string $original_directory;
    /** @var string $thumbnail_directory Cesta pro uložení náhledu. */
    protected string $thumbnail_directory;
    /** @var string $description Popis obrázku. */
    protected string $description;
    /** @var int $type Typ podle  https://www.php.net/manual/en/image.constants.php#constant.imagetype-gif PHP dokumentace */
    protected int $type;
    /** @var string Unikátní id vygenerované při vzniku objektu. */
    protected string $unique_id = "";
    /** @var int $orig_x Šířka originálního obrázku v px. */
    protected int $orig_x;
    /** @var int $orig_y Výška originálního obrkázku v px. */
    protected int $orig_y;
    /** @var int $th_x Šířka náhledu v px. */
    protected int $th_x;
    /** @var int $th_y Výška náhledu v px. */
    protected int $th_y;
    /** @var string $mime Detekovaný mime type. */
    protected string $mime;
    /** @var int $bits Detekovaná hloubka barev obrázku. */
    protected int $bits;

    /**
     * @param string $name Název souboru
     * @param string $description Popis obrázku
     * @param int $th_max_x Max šířka náhledu
     * @param int $th_max_y Max výška náhledu
     * @param string $origDir Adresář, kam se má uložit originál
     * @param string $thmbDir Adresář, kam se má uložit náhled
     */
    public function __construct(string $name, string $description, int $th_max_x = 100, int $th_max_y = 100, string $origDir, string $thmbDir)
    {
        $this->generateUniqueId();
        $this->name = $name;
        $this->description = $description;
        $this->th_x = $th_max_x;
        $this->th_y = $th_max_y;
        $this->original_directory = $origDir;
        $this->thumbnail_directory = $thmbDir;
    }

    private function generateUniqueId()
    {
        $this->unique_id = bin2hex(openssl_random_pseudo_bytes(16));
    }

    /**
     * Přiřadí k instaci skutečný obrázek z cesty source, uloží ho i jeho náhled.
     * @param string $source Soubor zdrojového obrázku.
     * @return void
     * @throws FileException Výjimka v případě, ze se nepodařilo soubor otevřít.
     * @throws ImageException Výjimka v případě, že soubor není obrázek.
     */
    public function generateImages(string $source)
    {
        if (!file_exists($source)) {
            throw new FileException("File $source not found.", $source);
        }
        $image_properties = @getimagesize($source);

        if ($image_properties) {
            $this->mime = $image_properties['mime'];
            $this->orig_x = $image_properties[0];
            $this->orig_y = $image_properties[1];
            $this->type = $image_properties[2];
            $this->bits = isset($image_properties['bits']) ? $image_properties['bits'] : 0;
        } else {
            throw new ImageException("File not recognized as an image", $source);
        }

        $this->createOriginal($source);
        $this->createThumbnail($source, $this->th_x, $this->th_y);
    }

    /**
     * Jednoduše zkopíruje obrázek do nového cíle, tj. do adresáře original. Název bude unikátní s použitím unikátního identifikátoru.
     * @param string $path Location of the original image.
     * @return void
     */
    protected function createOriginal(string $path): void
    {
        $new_name = $this->original_directory . DIRECTORY_SEPARATOR . $this->getUniqueName();
        copy($path, $new_name);
    }

    /**
     * Vyrobí náhled obrázku. Zde se používá knihovna GD k vyrobení nového obrázku a přenesení do něj zmenšeného originálu.
     * @param string $path Zdrojový obrázek.
     * @param int $max_x Max. šířka náhledu.
     * @param int $max_y Max. výška náhledu.
     * @return void
     * @throws ImageException Vyhodí výjimku v případě, že obrázek je neznámého typu.
     */
    protected function createThumbnail(string $path, int $max_x, int $max_y): void
    {
        $scale_x = min(1, $max_x / $this->orig_x);
        $scale_y = min(1, $max_y / $this->orig_y);
        $new_scale = min($scale_x, $scale_y);
        $this->th_x = $new_scale * $this->orig_x;
        $this->th_y = $new_scale * $this->orig_y;

        // vyrobime novy obrazek s novym rozmerem


        // nahrajeme obrazek ze zdroje
        // gd knihovna ma radu funkci, ktere nahravaji obrazek z ruznych formatu
        // jmenuji se vzdy imgcreatefromXXX, kde XXX je napr png, jpeg, gif apod
        // proto jsem si podle typu obrazku vyrobil metodu $this->getFunctionByImageType a ta mi vrati
        // spravny konec funkce, tj. to XXX. Kdyz to slozim s textem imgcreatefrom, tak dostanu
        // kandidata na tu funkci, kterou potrebuji.
        $thumb = imagecreatetruecolor($this->th_x, $this->th_y);
        $imgfn = $this->getFunctionNameByImageType("imagecreatefrom");
        if (!$imgfn || !function_exists($imgfn)) {
            throw new ImageException("Could not create a thumbnail for image $this->name", $path);
        }

        $source = $imgfn($path);
        // zkopirujeme ho v nove velikosti do cile
        $new_image = imagecopyresized($thumb, $source, 0, 0, 0, 0, $this->th_x, $this->th_y, $this->orig_x, $this->orig_y);
        // ulozime do souboru
        // jakou k tomu potrebujeme funkci?
        $save_fn = $this->getFunctionNameByImageType("image");
        $save_fn($thumb, $this->thumbnail_directory . DIRECTORY_SEPARATOR . $this->getUniqueName());
        imagedestroy($thumb); // uvolni pamet
        imagedestroy($source);
    }


    /**
     * Vrací název podle typu souboru.
     * @param string $fnPrefix Prefix, ktery je následně spojen s názvem typu obrázku.
     * @return string|false False se vrací tehdy, kdy se to nepovedlo.
     */
    private function getFunctionNameByImageType(string $fnPrefix): string|false
    {
        $ending = image_type_to_extension($this->type, false);
        if ($ending) {
            switch ($ending) {
                case "jpg":
                    $ending = "jpeg";
                    break;
            }
            return $fnPrefix . $ending;
        }
        return false;
    }

    // deletes it's files

    /**
     * Smaže soubory obrázku v lokacích original i náhledu.
     * @return void
     */
    public function destroy()
    {
        unlink($this->original_directory . DIRECTORY_SEPARATOR . $this->getUniqueName());
        unlink($this->thumbnail_directory . DIRECTORY_SEPARATOR . $this->getUniqueName());
    }

    /**
     * Vrací název souboru, pod kterým je uložen v original a náhledovém adresáři.
     * @return string
     */
    public function getUniqueName(): string
    {
        return $this->unique_id . $this->name;
    }

    /**
     * Vrací cestu k originálnímu souboru.
     * @return string Cestak originálu
     */
    public function getOriginalFullPath(): string
    {
        return $this->original_directory . DIRECTORY_SEPARATOR . $this->getUniqueName();
    }

    /**
     * Vrací cestu k náhledu včetně jména souboru náhledu.
     * @return string Cesta k náhledu.
     */
    public function getThumbnailFullPath(): string
    {
        return $this->thumbnail_directory . DIRECTORY_SEPARATOR . $this->getUniqueName();
    }

    public function getName(): string
    {
        return $this->name;
    }

    /**
     * Vrací šířku původního obrázku.
     * @return int Šířka původního obrázku.
     */
    public function getOrigX(): int
    {
        return $this->orig_x;
    }

    /**
     * Vrací výšku původního obrázku
     * @return int Výška původního obrázku.
     */
    public function getOrigY(): int
    {
        return $this->orig_y;
    }

    /**
     * Vrací popisek obrázku.
     * @return string Popisek obrázku.
     */
    public function getDescription(): string {
        return $this->description;
    }

}
