from neo4j import GraphDatabase
from typing import Dict, List, Optional

class MovieGraphApp:
    def __init__(self, uri="bolt://localhost:7687", user="user", password="password"):
        self.driver = GraphDatabase.driver(uri, auth=(user, password))

    def close(self):
        """Close the database connection."""
        self.driver.close()

    def verify_connectivity(self):
        """
        Verify connection to Neo4j database.
        
        Returns:
            bool: True if connection successful, False otherwise
        """
        try:
            self.driver.verify_connectivity()
            print("✓ Connected to Neo4j")
            return True
        except Exception as e:
            print(f"✗ Connection error: {str(e)}")
            return False

    def clear_database(self):
        """Remove all nodes and relationships from the database."""
        with self.driver.session() as session:
            query = """
                MATCH (n)
                DETACH DELETE n
            """
            session.run(query)
            print("✓ Database cleared")

    def create_node(self, label: str, properties: Dict):
        """
        Create a single node in the database.
        
        Args:
            label (str): Node label (e.g., 'ACTOR', 'MOVIE')
            properties (Dict): Node properties (e.g., name, year)
            
        Returns:
            Node: Created Neo4j node
        """
        with self.driver.session() as session:
            query = f"""
                CREATE (n:{label})
                SET n = $props
                RETURN n
            """
            result = session.run(query, props=properties)
            return result.single()["n"]

    def create_multiple_nodes(self, label: str, nodes_data: List[Dict]):
        """
        Create multiple nodes in a single transaction.
        
        Args:
            label (str): Node label for all nodes
            nodes_data (List[Dict]): List of property dictionaries for each node
        """
        with self.driver.session() as session:
            query = f"""
                UNWIND $nodes as node_data
                CREATE (n:{label})
                SET n = node_data
            """
            session.run(query, nodes=nodes_data)

    def create_actors(self):
        """Create initial set of actor nodes with predefined data."""
        actors = [
            {"id": "trojan", "name": "Ivan Trojan", "year": 1964},
            {"id": "machacek", "name": "Jiří Macháček", "year": 1966},
            {"id": "schneiderova", "name": "Jitka Schneiderová", "year": 1973},
            {"id": "sverak", "name": "Zdeněk Svěrák", "year": 1936}
        ]
        self.create_multiple_nodes("ACTOR", actors)
        print("✓ Actors created")

    def create_movies(self):
        """Create initial set of movie nodes with predefined data."""
        movies = [
            {"id": "samotari", "title": "Samotáři", "year": 2000},
            {"id": "medvidek", "title": "Medvídek", "year": 2007},
            {"id": "vratnelahve", "title": "Vratné lahve", "year": 2006}
        ]
        self.create_multiple_nodes("MOVIE", movies)
        print("✓ Movies created")

    def create_relationship(self, start_id: str, end_id: str, rel_type: str, 
                          properties: Optional[Dict] = None):
        """
        Create a single relationship between two nodes.
        
        Args:
            start_id (str): ID of the starting node
            end_id (str): ID of the ending node
            rel_type (str): Type of relationship (e.g., 'KNOWS', 'PLAY')
            properties (Dict, optional): Properties for the relationship
        """
        with self.driver.session() as session:
            query = f"""
                MATCH (a {{id: $start_id}}), (b {{id: $end_id}})
                CREATE (a)-[r:{rel_type}]->(b)
                SET r = $props
            """
            session.run(query, start_id=start_id, end_id=end_id, props=properties or {})

    def create_multiple_relationships(self, relationships: List[Dict], rel_type: str):
        """
        Create multiple relationships in a single transaction.
        
        Args:
            relationships (List[Dict]): List of relationship data
            rel_type (str): Type of relationship to create
        """
        formatted_rels = [
            {
                "start_id": rel["from"],
                "end_id": rel["to"],
                "properties": rel["properties"]
            }
            for rel in relationships
        ]
        
        with self.driver.session() as session:
            query = f"""
                UNWIND $rels as rel
                MATCH (a {{id: rel.start_id}}), (b {{id: rel.end_id}})
                CREATE (a)-[r:{rel_type}]->(b)
                SET r = rel.properties
            """
            session.run(query, rels=formatted_rels)

    def create_initial_relationships(self):
        """Create predefined set of KNOWS and PLAY relationships."""
        # Define actor relationships (who knows who)
        knows_relationships = [
            {"from": "trojan", "to": "machacek", "properties": {}},
            {"from": "trojan", "to": "schneiderova", "properties": {}},
            {"from": "machacek", "to": "schneiderova", "properties": {}},
            {"from": "sverak", "to": "machacek", "properties": {}}
        ]
        
        # Define movie-actor relationships (who plays in what)
        play_relationships = [
            {"from": "samotari", "to": "trojan", "properties": {}},
            {"from": "samotari", "to": "machacek", "properties": {}},
            {"from": "samotari", "to": "schneiderova", "properties": {}},
            {"from": "medvidek", "to": "trojan", "properties": {}},
            {"from": "vratnelahve", "to": "sverak", "properties": {}}
        ]

        self.create_multiple_relationships(knows_relationships, "KNOWS")
        self.create_multiple_relationships(play_relationships, "PLAY")
        print("✓ Relationships created")

    def update_node(self, label: str, node_id: str, updates: Dict):
        """
        Update properties of an existing node.
        
        Args:
            label (str): Node label
            node_id (str): ID of node to update
            updates (Dict): Properties to update
        """
        with self.driver.session() as session:
            query = f"""
                MATCH (n:{label} {{id: $node_id}})
                SET n += $updates
            """
            session.run(query, node_id=node_id, updates=updates)

    def update_relationship(self, start_id: str, end_id: str, 
                          rel_type: str, updates: Dict):
        """
        Update properties of an existing relationship.
        
        Args:
            start_id (str): ID of start node
            end_id (str): ID of end node
            rel_type (str): Type of relationship
            updates (Dict): Properties to update
        """
        with self.driver.session() as session:
            query = f"""
                MATCH (a {{id: $start_id}})-[r:{rel_type}]->(b {{id: $end_id}})
                SET r += $updates
            """
            session.run(query, start_id=start_id, end_id=end_id, updates=updates)

    def delete_node(self, node_id: str):
        """
        Delete a node and all its relationships.
        
        Args:
            node_id (str): ID of node to delete
        """
        with self.driver.session() as session:
            query = """
                MATCH (n {id: $node_id})
                DETACH DELETE n
            """
            session.run(query, node_id=node_id)

    def delete_relationship(self, start_id: str, end_id: str, rel_type: str):
        """
        Delete a specific relationship between two nodes.
        
        Args:
            start_id (str): ID of start node
            end_id (str): ID of end node
            rel_type (str): Type of relationship to delete
        """
        with self.driver.session() as session:
            query = f"""
                MATCH (a {{id: $start_id}})-[r:{rel_type}]->(b {{id: $end_id}})
                DELETE r
            """
            session.run(query, start_id=start_id, end_id=end_id)

    def find_actor_movies(self, actor_id: str):
        """
        Find all movies an actor has played in.
        
        Args:
            actor_id (str): ID of the actor
            
        Returns:
            List[Tuple]: List of (movie_title, year) tuples
        """
        with self.driver.session() as session:
            query = """
                MATCH (a:ACTOR {id: $actor_id})<-[:PLAY]-(m:MOVIE)
                RETURN m.title as movie, m.year as year
                ORDER BY m.year
            """
            result = session.run(query, actor_id=actor_id)
            return [(r["movie"], r["year"]) for r in result]

    def find_movie_cast(self, movie_id: str):
        """
        Find all actors in a specific movie.
        
        Args:
            movie_id (str): ID of the movie
            
        Returns:
            List[Tuple]: List of (actor_name, birth_year) tuples
        """
        with self.driver.session() as session:
            query = """
                MATCH (m:MOVIE {id: $movie_id})-[:PLAY]->(a:ACTOR)
                RETURN a.name as name, a.year as birth_year
                ORDER BY a.name
            """
            result = session.run(query, movie_id=movie_id)
            return [(r["name"], r["birth_year"]) for r in result]

    def find_actor_collaborations(self, actor_id: str):
        """
        Find all actors who have worked with a given actor.
        
        Args:
            actor_id (str): ID of the actor
            
        Returns:
            List[Tuple]: List of (actor_name, movies_list) tuples
        """
        with self.driver.session() as session:
            query = """
                MATCH (m:MOVIE)-[:PLAY]->(a1:ACTOR {id: $actor_id})
                MATCH (m)-[:PLAY]->(a2:ACTOR)
                WHERE a1 <> a2
                RETURN DISTINCT a2.name as name, collect(m.title) as movies
                ORDER BY size(collect(m.title)) DESC
            """
            result = session.run(query, actor_id=actor_id)
            return [(r["name"], r["movies"]) for r in result]

    def analyze_actor_network(self, actor_id: str):
        """
        Analyze an actor's collaboration network up to 2 steps away.
        
        Args:
            actor_id (str): ID of the actor
            
        Returns:
            List[Tuple]: List of (actor_name, distance, paths_count) tuples
        """
        with self.driver.session() as session:
            query = """
                MATCH (center:ACTOR {id: $actor_id})
                MATCH (connected:ACTOR)
                WHERE center <> connected
                WITH center, connected
                MATCH p = shortestPath((center)-[*1..2]-(connected))
                RETURN connected.name as name,
                       length(p) as distance,
                       count(*) as paths
                ORDER BY distance, paths DESC
            """
            result = session.run(query, actor_id=actor_id)
            return [(r["name"], r["distance"], r["paths"]) 
                    for r in result]

    def display_medvidek_info(self):
        """
        Display detailed information about Medvídek movie actors and their network.
        Shows direct connections (friends) and secondary connections (friends of friends).
        """
        with self.driver.session() as session:
            query = """
                MATCH (m:MOVIE {id: 'medvidek'})-[r:PLAY]->(a:ACTOR)
                WITH a
                OPTIONAL MATCH (a)-[:KNOWS]-(friend:ACTOR)
                OPTIONAL MATCH (friend)-[:KNOWS]-(friend_of_friend:ACTOR)
                WHERE friend_of_friend <> a
                RETURN 
                    a.name as actor,
                    collect(DISTINCT friend.name) as direct_friends,
                    collect(DISTINCT friend_of_friend.name) as friends_of_friends
            """
            result = session.run(query)
            records = list(result)
            
            if not records:
                print("\nNo actors found for Medvídek!")
                return
                
            print("\nMedvídek actors and their connections:")
            for record in records:
                actor = record['actor']
                direct_friends = record['direct_friends']
                friends_of_friends = record['friends_of_friends']
                
                print(f"\n{actor}:")
                if direct_friends:
                    print("Direct friends:")
                    for friend in direct_friends:
                        print(f"- {friend}")
                if friends_of_friends:
                    print("Friends of friends:")
                    for fof in friends_of_friends:
                        if fof not in direct_friends:  # Avoid duplicates
                            print(f"- {fof}")
                if not direct_friends and not friends_of_friends:
                    print("No connections found")

    def find_all_movies(self):
        """
        Retrieve all movies from the database.
        
        Returns:
            List[Tuple]: List of (movie_title, year) tuples
        """
        with self.driver.session() as session:
            query = """
                MATCH (m:MOVIE)
                RETURN DISTINCT m.title as title, m.year as year
                ORDER BY m.year
            """
            result = session.run(query)
            return [(r["title"], r["year"]) for r in result]

def main():
    """
    Main function to demonstrate the MovieGraphApp functionality.
    Creates a sample movie-actor graph and runs various queries on it.
    """
    app = MovieGraphApp()
    try:
        # Verify database connection
        if not app.verify_connectivity():
            return
            
        # Initialize database
        app.clear_database()
        print("\nCreating initial graph...")
        app.create_actors()
        app.create_movies()
        app.create_initial_relationships()

        # Run basic queries
        print("\nQuerying actor information...")
        movies = app.find_actor_movies("trojan")
        print("\nIvan Trojan's movies:")
        for movie, year in movies:
            print(f"- {movie} ({year})")

        # Run advanced queries
        print("\nAnalyzing collaborations...")
        collaborations = app.find_actor_collaborations("trojan")
        print("\nIvan Trojan's collaborations:")
        for actor, movies in collaborations:
            print(f"- {actor}: {', '.join(movies)}")

        # Run network analysis
        print("\nAnalyzing network...")
        network = app.analyze_actor_network("trojan")
        print("\nIvan Trojan's network:")
        for name, distance, paths in network:
            print(f"- {name}: {distance} step(s) away, {paths} connection(s)")

        # Display special movie info
        app.display_medvidek_info()

        # Display all movies
        print("\nAll movies in the database:")
        all_movies = app.find_all_movies()
        for title, year in all_movies:
            print(f"- {title} ({year})")

    finally:
        # Ensure connection is closed even if an error occurs
        app.close()

if __name__ == "__main__":
    main()
