from neo4j import GraphDatabase
from enum import Enum

# Connection configuration
URI = "bolt://localhost:7687"
USER = "user"
PASSWORD = "password"  # Change to your password

class RelTypes(Enum):
    KNOWS = "KNOWS"
    PLAY = "PLAY"

class MovieGraphApp:
    def __init__(self):
        self.driver = GraphDatabase.driver(URI, auth=(USER, PASSWORD))

    def verify_connectivity(self):
        try:
            self.driver.verify_connectivity()
            print("✓ Successfully connected to Neo4j")
            return True
        except Exception as e:
            print(f"✗ Connection error: {str(e)}")
            return False

    def close(self):
        self.driver.close()

    def clear_database(self):
        with self.driver.session() as session:
            session.run("MATCH (n) DETACH DELETE n")
            print("✓ Database cleared")

    def create_graph(self):
        with self.driver.session() as session:
            # Creating nodes and relationships in a single query
            query = """
            CREATE (actor1:ACTOR {id: 'trojan', name: 'Ivan Trojan', year: 1964}),
                   (actor2:ACTOR {id: 'machacek', name: 'Jiří Macháček', year: 1966}),
                   (actor3:ACTOR {id: 'schneiderova', name: 'Jitka Schneiderová', year: 1973}),
                   (actor4:ACTOR {id: 'sverak', name: 'Zdeněk Svěrák', year: 1936}),
                   
                   (movie1:MOVIE {id: 'samotari', title: 'Samotáři', year: 2000}),
                   (movie2:MOVIE {id: 'medvidek', title: 'Medvídek', year: 2007}),
                   (movie3:MOVIE {id: 'vratnelahve', title: 'Vratné lahve', year: 2006}),
                   
                   (actor1)-[:KNOWS]->(actor2),
                   (actor1)-[:KNOWS]->(actor3),
                   (actor2)-[:KNOWS]->(actor1),
                   (actor2)-[:KNOWS]->(actor3),
                   (actor4)-[:KNOWS]->(actor2),
                   
                   (movie1)-[:PLAY]->(actor1),
                   (movie1)-[:PLAY]->(actor2),
                   (movie1)-[:PLAY]->(actor3),
                   (movie2)-[:PLAY]->(actor1),
                   (movie3)-[:PLAY]->(actor4)
            """
            session.run(query)
            print("✓ Graph created")

    def traversal_1(self):
        with self.driver.session() as session:
            result = session.run("""
                MATCH (start:ACTOR {id: 'trojan'})
                MATCH (start)-[:KNOWS*1..]->(connected:ACTOR)
                RETURN DISTINCT connected.name as name
            """)
            print("\nTRAVERSAL #1 (Actors connected through KNOWS relationship):")
            for record in result:
                print(record["name"])

    def traversal_2(self):
        with self.driver.session() as session:
            result = session.run("""
                MATCH (start:MOVIE {id: 'medvidek'})
                MATCH path = (start)-[:PLAY|KNOWS*1..]->(connected)
                WHERE NOT connected:MOVIE
                RETURN DISTINCT connected.name as name
            """)
            print("\nTRAVERSAL #2 (Actors connected to Medvídek movie):")
            for record in result:
                print(record["name"])

    # Additional queries for data manipulation and retrieval
    
    def add_new_actor(self, actor_id, name, year):
        """Add a new actor to the graph"""
        with self.driver.session() as session:
            session.run("""
                CREATE (a:ACTOR {id: $id, name: $name, year: $year})
                """, id=actor_id, name=name, year=year)
            print(f"✓ Added new actor: {name}")

    def add_movie_relationship(self, actor_id, movie_id):
        """Create PLAY relationship between movie and actor"""
        with self.driver.session() as session:
            session.run("""
                MATCH (m:MOVIE {id: $movie_id})
                MATCH (a:ACTOR {id: $actor_id})
                CREATE (m)-[:PLAY]->(a)
                """, movie_id=movie_id, actor_id=actor_id)
            print(f"✓ Added relationship: {movie_id} -> {actor_id}")

    def update_movie_year(self, movie_id, new_year):
        """Update movie's release year"""
        with self.driver.session() as session:
            session.run("""
                MATCH (m:MOVIE {id: $movie_id})
                SET m.year = $new_year
                """, movie_id=movie_id, new_year=new_year)
            print(f"✓ Updated movie year: {movie_id} -> {new_year}")

    def get_actors_by_year_range(self, start_year, end_year):
        """Find actors born between specified years"""
        with self.driver.session() as session:
            result = session.run("""
                MATCH (a:ACTOR)
                WHERE a.year >= $start_year AND a.year <= $end_year
                RETURN a.name as name, a.year as year
                ORDER BY a.year
                """, start_year=start_year, end_year=end_year)
            print(f"\nActors born between {start_year} and {end_year}:")
            for record in result:
                print(f"{record['name']} ({record['year']})")

    def get_movie_statistics(self):
        """Get statistics about movies and actors"""
        with self.driver.session() as session:
            result = session.run("""
                MATCH (m:MOVIE)
                OPTIONAL MATCH (m)-[:PLAY]->(a:ACTOR)
                WITH m, count(a) as actor_count
                RETURN m.title as title, 
                       m.year as year,
                       actor_count as number_of_actors
                ORDER BY m.year DESC
                """)
            print("\nMovie Statistics:")
            for record in result:
                print(f"{record['title']} ({record['year']}) - {record['number_of_actors']} actors")

    def find_common_movies(self, actor1_id, actor2_id):
        """Find movies where both actors played"""
        with self.driver.session() as session:
            result = session.run("""
                MATCH (a1:ACTOR {id: $actor1_id})<-[:PLAY]-(m:MOVIE)-[:PLAY]->(a2:ACTOR {id: $actor2_id})
                RETURN m.title as movie, m.year as year
                """, actor1_id=actor1_id, actor2_id=actor2_id)
            print(f"\nCommon movies between actors {actor1_id} and {actor2_id}:")
            if not result.peek():
                print("No common movies found")
            for record in result:
                print(f"{record['movie']} ({record['year']})")

    def find_movie_actors_and_friends(self, movie_id):
        with self.driver.session() as session:
            result = session.run("""
                // Start with the movie
                MATCH (m:MOVIE {id: $movie_id})
                // Find actors in this movie and their friends up to 2 levels deep
                MATCH (m)-[:PLAY]->(actor:ACTOR)-[:KNOWS*0..2]-(friend:ACTOR)
                RETURN DISTINCT friend.name as name
                ORDER BY friend.name
            """, movie_id=movie_id)
            
            print(f"\nActors in {movie_id} and their network:")
            return [record["name"] for record in result]

def main():
    app = MovieGraphApp()
    try:
        if not app.verify_connectivity():
            return
        
        # Clear and create initial graph
        app.clear_database()
        app.create_graph()
        
        # Basic traversals
        app.traversal_1()
        app.traversal_2()
        
        # Additional operations
        print("\n=== Performing additional operations ===")
        
        # Add new actor and create relationship
        app.add_new_actor('polakova', 'Anna Polakova', 1975)
        app.add_movie_relationship('polakova', 'medvidek')
        
        # Update movie year
        app.update_movie_year('medvidek', 2008)
        
        # Get various statistics
        app.get_actors_by_year_range(1960, 1970)
        app.get_movie_statistics()
        app.find_common_movies('trojan', 'machacek')

        # Actors who played in Medvídek movie
        actors = app.find_movie_actors_and_friends('medvidek')
        print("All connected actors:", actors)
        
    except Exception as e:
        print(f"✗ Error occurred: {str(e)}")
    finally:
        app.close()

if __name__ == "__main__":
    main()
