"""
HistogramVisualizer: Unified and flexible histogram plotting utility using matplotlib.

Author: Vojtěch Drahý
Copyright (c) 2025 Vojtěch Drahý. All rights reserved.

This module provides a single function `HistogramVisualizer` to create histograms with:
- linear or logarithmic axes,
- optional logarithmic bins,
- customizable figure size, colors, alpha, labels, and grid,
- ability to draw in existing subplots or create a new figure.

The function is compatible with both standalone scripts and Jupyter/VSCode notebooks.
"""

import matplotlib.pyplot as plt
import numpy as np

def HistogramVisualizer(x, 
              sPlotIndex=111, 
              bins=100, 
              log=False, 
              logBins=False, 
              logX=False, 
              logY=False,
              xLabel='', 
              yLabel='Count', 
              title=None, 
              isGrid=False,
              create_fig=False, 
              figsize=(4,3),
              node_color='green', 
              alpha=0.5, 
              density=False, 
              **kwargs):
    """
    Plot a histogram and return a dictionary of bin center -> frequency.

    Parameters
    ----------
    x : array-like
        Data to be histogrammed.
    sPlotIndex : int or tuple, optional
        Matplotlib subplot index (default=111). Tuple if using gridspec.
    bins : int or sequence, optional
        Number of bins or sequence of bin edges (default=100).
    log : bool, optional
        If True, the histogram is plotted with a log-scaled y-axis.
    logBins : bool, optional
        If True, use logarithmically spaced bins (default=False).
    logX : bool, optional
        If True, set the x-axis to logarithmic scale.
    logY : bool, optional
        If True, set the y-axis to logarithmic scale.
    xLabel : str, optional
        Label for the x-axis.
    yLabel : str, optional
        Label for the y-axis (default='Count').
    title : str, optional
        Title of the plot.
    isGrid : bool, optional
        Whether to display the grid (default=False).
    create_fig : bool, optional
        If True, create a new figure (default=False).
    figsize : tuple, optional
        Figure size if creating a new figure (width, height in inches).
    node_color : str or list, optional
        Color(s) of the histogram bars.
    alpha : float, optional
        Transparency of the bars (0.0 transparent, 1.0 opaque).
    density : bool, optional
        If True, normalize the histogram (default=False).
    **kwargs :
        Additional keyword arguments passed to `matplotlib.pyplot.hist`.

    Returns
    -------
    bin_dict : dict
        Dictionary where keys are bin centers and values are counts (or densities if density=True).
    """
    
    if create_fig:
        plt.figure(figsize=figsize)
    
    # Select subplot
    if isinstance(sPlotIndex, tuple):
        ax = plt.subplot(*sPlotIndex)
    else:
        ax = plt.subplot(sPlotIndex)
    
    # Handle logarithmic bins
    if logBins:
        positive_values = [v for v in x if v > 0]
        if len(positive_values) == 0:
            raise ValueError("Data must contain positive values for logBins=True.")
        bins = np.logspace(np.log10(min(positive_values)/2), np.log10(max(positive_values)), num=bins)
    
    # Plot histogram
    counts, bin_edges, patches = ax.hist(
        x, 
        bins=bins, 
        density=density, 
        log=log, 
        facecolor=node_color, 
        alpha=alpha,
        **kwargs
    )
    
    # Set axes scales
    if logX: ax.set_xscale('log')
    if logY: ax.set_yscale('log', nonposy='clip')
    
    # Set labels and title
    if xLabel: ax.set_xlabel(xLabel)
    if yLabel: ax.set_ylabel(yLabel)
    if title: ax.set_title(title)
    
    # Grid
    if isGrid: ax.grid(isGrid)
    
    # Show plot
    plt.show()

    # Build bin center -> count dictionary
    bin_centers = (bin_edges[:-1] + bin_edges[1:]) / 2
    bin_dict = {center: count for center, count in zip(bin_centers, counts)}
    
    return bin_dict
