#include "FABRIKComponent.h"

#include "Kismet/KismetMathLibrary.h"

// Sets default values for this component's properties
UFABRIKComponent::UFABRIKComponent()
{
	// Set this component to be initialized when the game starts, and to be ticked every frame.  You can turn these features
	// off to improve performance if you don't need them.
	PrimaryComponentTick.bCanEverTick = true;
}

void UFABRIKComponent::FABRIKSingleIteration()
{
	// All FABRIK parameters are class variables: BoneChainIndex, Target, PoseableMesh
	// BoneChain contains names which can be used with PoseableMesh's methods and original distances between bones in world space.
	// Index 0 is the root bone of the chain, last index is the effector/end bone.
	// Target is an actor pointer which is guaranteed to exist here.
	// This method should assume Target is reachable and do a single iteration of FABRIK, both the forward
	// and the backward reaching steps (no need to check distance to target).

	int Root = 0;
	int Effector = BoneChain.Num() - 1;
	
	FVector TargetLocation = Target->GetActorLocation();
	FVector RootLocation = PoseableMesh->GetBoneLocationByName(BoneChain[Root].BoneName, EBoneSpaces::WorldSpace);

	// Forward reaching stage
	PoseableMesh->SetBoneLocationByName(BoneChain[Effector].BoneName, TargetLocation, EBoneSpaces::WorldSpace);
	for (int i = Effector - 1; i >= Root; --i)
	{
		FVector Location = PoseableMesh->GetBoneLocationByName(BoneChain[i].BoneName, EBoneSpaces::WorldSpace);
		FVector LocationNext = PoseableMesh->GetBoneLocationByName(BoneChain[i + 1].BoneName, EBoneSpaces::WorldSpace);
		
		double Distance = FVector::Distance(Location, LocationNext);
		double Alpha = BoneChain[i].BoneDistance / Distance;

		const FVector NewLocation = (1 - Alpha) * LocationNext + Alpha * Location;
		PoseableMesh->SetBoneLocationByName(BoneChain[i].BoneName, NewLocation, EBoneSpaces::WorldSpace);
	}

	// Backward reaching stage
	PoseableMesh->SetBoneLocationByName(BoneChain[Root].BoneName, RootLocation, EBoneSpaces::WorldSpace);
	for (int i = Root; i < Effector; ++i)
	{
		FVector Location = PoseableMesh->GetBoneLocationByName(BoneChain[i].BoneName, EBoneSpaces::WorldSpace);
		FVector LocationNext = PoseableMesh->GetBoneLocationByName(BoneChain[i + 1].BoneName, EBoneSpaces::WorldSpace);
		
		double Distance = FVector::Distance(Location, LocationNext);
		double Alpha = BoneChain[i].BoneDistance / Distance;

		const FVector NewLocation = (1 - Alpha) * Location + Alpha * LocationNext;
		PoseableMesh->SetBoneLocationByName(BoneChain[i + 1].BoneName, NewLocation, EBoneSpaces::WorldSpace);
	}

	FixRotations();
}


// Called when the game starts
void UFABRIKComponent::BeginPlay()
{
	Super::BeginPlay();

	// Get Poseable mesh first - if we don't have this, we can't do IK!
	PoseableMesh = GetOwner()->FindComponentByClass<UPoseableMeshComponent>();
	if (!PoseableMesh)
	{
		UE_LOG(LogClass, Error, TEXT("No PoseableMeshComponent found!"));
		return;
	}

	if (BoneChainEnd.IsNone())
	{
		UE_LOG(LogClass, Error, TEXT("BoneChainEnd has to be specified and has to be a bone name!"));
		return;
	}

	// Gather bone names and their lengths (initial distances from each other) by traversing parent bones.
	BoneChain.Add({BoneChainEnd, 0});
	for (int i = 0; i < BoneChainLength; ++i)
	{
		const FName Child = BoneChain.Last(0).BoneName;
		const FName Parent = PoseableMesh->GetParentBone(Child);

		if (Parent.IsNone())
		{
			UE_LOG(LogClass, Warning,
			       TEXT("BoneChainEnd doesn't belong to a long enough chain! Chain will only be %d bones long."),
			       BoneChain.Num());
			break;
		}

		const double Distance = FVector::Distance(
			PoseableMesh->GetBoneLocationByName(Child, EBoneSpaces::WorldSpace),
			PoseableMesh->GetBoneLocationByName(Parent, EBoneSpaces::WorldSpace)
		);

		BoneChain.Add({Parent, Distance});
	}
	
	// Swap to align bone order with the paper
	for (int i = 0; i < BoneChain.Num() / 2; ++i)
	{
		Swap(BoneChain[i], BoneChain[BoneChain.Num() - i - 1]);
	}
}

bool UFABRIKComponent::IsTargetReachable()
{
	if (BoneChain.IsEmpty())
		return false;
	
	double Sum = 0;
	for (int i = 0; i < BoneChain.Num(); ++i)
	{
		Sum += BoneChain[i].BoneDistance;
	}

	return FVector::Distance(Target->GetActorLocation(),
	                         PoseableMesh->GetBoneLocationByName(BoneChain[0].BoneName, EBoneSpaces::WorldSpace)) < Sum;
}

void UFABRIKComponent::FixRotations()
{
	TArray<FVector> Locations;
	// Save locations as modifying rotations will change locations as well.
	for (int i = 0; i < BoneChain.Num(); ++i)
	{
		Locations.Add(PoseableMesh->GetBoneLocationByName(BoneChain[i].BoneName, EBoneSpaces::WorldSpace));
	}

	// Modify rotation of each bone to point to the next bone in world space.
	const FVector TargetLocation = Target->GetActorLocation();
	for (int i = 0; i < BoneChain.Num() - 1; ++i)
	{
		FVector Location = Locations[i];
		FVector LocationNext = Locations[i + 1];
		FRotator NewRotation = UKismetMathLibrary::MakeRotFromY(Location - LocationNext);
		
		PoseableMesh->SetBoneRotationByName(BoneChain[i].BoneName, NewRotation, EBoneSpaces::WorldSpace);
	}

	// Recall saved locations.
	for (int i = 0; i < BoneChain.Num(); ++i)
	{
		PoseableMesh->SetBoneLocationByName(BoneChain[i].BoneName, Locations[i], EBoneSpaces::WorldSpace);
		const FVector Location = PoseableMesh->GetBoneLocationByName(BoneChain[i].BoneName, EBoneSpaces::WorldSpace);
	}
}


// Called every frame
void UFABRIKComponent::TickComponent(float DeltaTime, ELevelTick TickType,
                                     FActorComponentTickFunction* ThisTickFunction)
{
	Super::TickComponent(DeltaTime, TickType, ThisTickFunction);

	if (EnableFABRIK && Target)
	{
		// Small optimization as per the FABRIK paper in case we cannot reach the target - just position
		// all bones in a straight line.
		if (!IsTargetReachable())
		{
			const FVector TargetLocation = Target->GetActorLocation();
			for (int i = 0; i < BoneChain.Num() - 1; ++i)
			{
				FVector Location = PoseableMesh->GetBoneLocationByName(BoneChain[i].BoneName, EBoneSpaces::WorldSpace);

				const double Distance = FVector::Distance(TargetLocation, Location);
				const double Alpha = BoneChain[i].BoneDistance / Distance;

				const FVector NewLocation = (1 - Alpha) * Location + Alpha * TargetLocation;
				PoseableMesh->SetBoneLocationByName(BoneChain[i + 1].BoneName, NewLocation, EBoneSpaces::WorldSpace);
			}

			FixRotations();
		}
		else
		{
			// We can reach the target, so start doing FABRIK iterations.
			for (int i = 0; i < MaxIterations; ++i)
			{
				double Distance = FVector::Distance(
					Target->GetActorLocation(),
					PoseableMesh->GetBoneLocationByName(BoneChain[0].BoneName, EBoneSpaces::WorldSpace));
				if (Distance < TargetDistanceTolerance)
				{
					break;
				}

				FABRIKSingleIteration();
			}
		}

		PoseableMesh->RefreshBoneTransforms();
	}
}
