#include "ClothSimulationComponent.h"

DEFINE_LOG_CATEGORY(ClothSimulationComponent);

// Sets default values for this component's properties
UClothSimulationComponent::UClothSimulationComponent()
{
	// Set this component to be initialized when the game starts, and to be ticked every frame.  You can turn these features
	// off to improve performance if you don't need them.
	PrimaryComponentTick.bCanEverTick = true;

	// ...
}


void UClothSimulationComponent::StartConstantTimer()
{
	GetWorld()->GetTimerManager().SetTimer(TimerHandle, [this]()
	{
		TickSimulation(ConstantTimeStep);
	}, ConstantTimeStep, true, ConstantTimeStep);
}

// Called when the game starts
void UClothSimulationComponent::BeginPlay()
{
	Super::BeginPlay();

	if (bUseConstantTimeStep)
	{
		StartConstantTimer();
	}
}


// Called every frame
void UClothSimulationComponent::TickComponent(float DeltaTime, ELevelTick TickType, FActorComponentTickFunction* ThisTickFunction)
{
	Super::TickComponent(DeltaTime, TickType, ThisTickFunction);
	if (!bUseConstantTimeStep)
	{
		float Delta = DeltaTime * TimeScale;

		TickSimulation(Delta);
	}
}

bool UClothSimulationComponent::HasValidParameters()
{
	return GridSize > 3;
}

void UClothSimulationComponent::GenerateCloth()
{
	if (HasValidParameters())
	{
		ClothNodes.Empty();
		ClothNodes.Reserve(GridSize * GridSize);
		const FVector2d SectionSize = FVector2d(Scale / GridSize, Scale / GridSize);
		
		// Create cloth nodes (mass points) in a downward-facing grid
		for (int32 Y = 0; Y < GridSize; ++Y)
		{
			for (int32 X = 0; X < GridSize; ++X)
			{
				FVector NodePosition(0.f, Y * SectionSize.Y, X * -SectionSize.X);
				ClothNodes.Emplace(MakeShared<FClothNode>(NodePosition, NodeMass));
			}
		}

		// Fix two corner points
		ClothNodes[0]->SetFixed(true);
		ClothNodes[ClothNodes.Num() - GridSize]->SetFixed(true);
		
		// Add springs to nodes
		ClothSprings.Empty();

		// structural springs
		//
		//  o    o    o
		//       |
		//  o -- o -- o
		//       |
		//  o    o    o
		//
		for (int i = 0; i < GridSize; ++i)
		{
			for (int j = 1; j < GridSize; ++j)
			{
				FClothNodeRef First = ClothNodes[NodeIndex(i, j)];
				FClothNodeRef Second = ClothNodes[NodeIndex(i, j - 1)];

				ClothSprings.Add(MakeShared<FClothSpring>(First, Second, StructuralElasticity, Damping));

				First = ClothNodes[NodeIndex(j, i)];
				Second = ClothNodes[NodeIndex(j - 1, i)];

				ClothSprings.Add(MakeShared<FClothSpring>(First, Second, StructuralElasticity, Damping));
			}
		}

		// shear springs
		//
		//  o    o    o
		//     \   /  
		//  o    o    o
		//     /   \
		//  o    o    o
		//
		for (int i = 1; i < GridSize; ++i)
		{
			for (int j = 1; j < GridSize; ++j)
			{
				FClothNodeRef First = ClothNodes[NodeIndex(i, j)];
				FClothNodeRef Second = ClothNodes[NodeIndex(i - 1, j - 1)];

				ClothSprings.Add(MakeShared<FClothSpring>(First, Second, ShearElasticity, Damping));

				First = ClothNodes[NodeIndex(GridSize - i - 1, j)];
				Second = ClothNodes[NodeIndex(GridSize - i, j - 1)];

				ClothSprings.Add(MakeShared<FClothSpring>(First, Second, ShearElasticity, Damping));
			}
		}

		// bend springs (over one)
		//
		//  o    o    o    o    o
		//            |
		//  o    o    x    o    o
		//            |
		//  o----x----o----x----o
		//            |
		//  o    o    x    o    o
		//            |
		//  o    o    o    o    o
		//
		for (int i = 0; i < GridSize; ++i)
		{
			for (int j = 2; j < GridSize; ++j)
			{
				FClothNodeRef First = ClothNodes[NodeIndex(i, j)];
				FClothNodeRef Second = ClothNodes[NodeIndex(i, j - 2)];

				ClothSprings.Add(MakeShared<FClothSpring>(First, Second, BendElasticity, Damping));

				First = ClothNodes[NodeIndex(j, i)];
				Second = ClothNodes[NodeIndex(j - 2, i)];

				ClothSprings.Add(MakeShared<FClothSpring>(First, Second, BendElasticity, Damping));
			}
		}
	}
	else
	{
		UE_LOG(ClothSimulationComponent, Error, TEXT("Invalid cloth simulation parameters!"));
	}
}

FVector UClothSimulationComponent::GetNodePosition(int32 NodeIndex) const
{
	return ClothNodes[NodeIndex]->GetPosition();
}

void UClothSimulationComponent::TickSimulation(float DeltaTime)
{
	const FTransform Transform = GetOwner()->GetActorTransform();

	UWorld* World = GetWorld();
	FVector Gravity = FVector::ZAxisVector * World->GetGravityZ() * DeltaTime;

	// First recalculate new node positions based on their current calculated forces.
	for (FClothNodeRef Node : ClothNodes)
	{
		Node->Tick(DeltaTime, World, Transform);
	}
	
	// Then clear acting forces on each node, add gravity and wind
	for (FClothNodeRef Node : ClothNodes)
	{
		Node->ClearForce();
		Node->AddForce(Gravity);
		Node->AddForce(WindForce);
	}

	// Finally add spring forces (through the actual springs, as each spring
	// knows which two nodes it's connected to).
	for (FClothSpringRef Spring : ClothSprings)
	{
		Spring->AddForceToNodes();
	}
}

int UClothSimulationComponent::NodeIndex(int X, int Y) const
{
	return GridSize * Y + X;
}

