#include "ClothNode.h"
#include "ClothInteractionObject.h"
#include "EngineUtils.h"

void FClothNode::Tick(float DeltaTime, UWorld* World, const FTransform& ParentTransform)
{
	const FVector LastPosition = Position;

	if (bFixed)
	{
		// Fixed points will stay still, so just symbolically zero out the speed
		Speed = FVector::ZeroVector;
	}
	else
	{
		// Simple differential time step
		const FVector Acceleration = ActingForce / Mass;
		Speed += DeltaTime * Acceleration;
		Position += DeltaTime * Speed;
	}

	// Checks for collision only with our player ball actor.
	// Collision with other objects could be more easily achieved by i.e. each
	// cloth node being its own actor (or more accurately, a collider or a rigid body),
	// so that they can participate in Unreal's physics calculations
	// more easily (as a kinematic object).
	FRotator BackRotator = ParentTransform.Rotator().GetInverse();
	for (TActorIterator<AActor> it(World); it; ++it) {
		if (it->GetClass()->ImplementsInterface(UClothInteractionObject::StaticClass()))
		{
			FVector Location = IClothInteractionObject::Execute_GetWorldPosition(*it);
			Location = BackRotator.RotateVector(Location - ParentTransform.GetLocation());
			float SphereRadius = IClothInteractionObject::Execute_GetSphereRadius(*it);
			if (TestSphereIntersection(Location, SphereRadius)) {
				FVector ToNode = LastPosition - Location;
				ChangeSpeedAccordingToIntersectedSphere(Location, LastPosition);
				ToNode.Normalize();
				ToNode = ToNode * SphereRadius * 1.05f;
				Position = Location + ToNode;
				Position += Speed * DeltaTime;
			}
		}
	}
}

void FClothNode::SetFixed(bool bInFixed)
{
	bFixed = bInFixed;
}

void FClothNode::ClearForce()
{
	ActingForce = FVector::ZeroVector;
}

void FClothNode::AddForce(const FVector& InForce)
{
	ActingForce += InForce;
}

FVector FClothNode::GetPosition() const
{
	return Position;
}

FVector FClothNode::GetSpeed() const
{
	return Speed;
}

void FClothNode::ChangeSpeedAccordingToIntersectedSphere(const FVector& IntersectionPoint,
	const FVector& NodePosition)
{
	FVector ToNode = NodePosition - IntersectionPoint;
	ToNode.Normalize();
	Speed = FVector::VectorPlaneProject(Speed, ToNode);
}

bool FClothNode::TestSphereIntersection(const FVector& Location, float Radius) const
{
	return FVector::DistSquared(Location, Position) < Radius * Radius * 1.05f;
}
