#include "ClothActor.h"

#include "RealtimeMeshSimple.h"


// Sets default values
AClothActor::AClothActor()
{
 	// Set this actor to call Tick() every frame.  You can turn this off to improve performance if you don't need it.
	PrimaryActorTick.bCanEverTick = true;

	ClothSimulation = CreateDefaultSubobject<UClothSimulationComponent>(TEXT("Cloth Simulation"));
}

// Called when the game starts or when spawned
void AClothActor::BeginPlay()
{
	Super::BeginPlay();
	
	ClothSimulation->NodeMass = NodeMass;
	ClothSimulation->StructuralElasticity = StructuralElasticity;
	ClothSimulation->ShearElasticity = ShearElasticity;
	ClothSimulation->BendElasticity = BendElasticity;
	ClothSimulation->Damping = Damping;
	ClothSimulation->TimeScale = TimeScale;
	ClothSimulation->Scale = Scale;
	ClothSimulation->GridSize = GridSize;
	ClothSimulation->bUseConstantTimeStep = bUseConstantTimeStep;
	ClothSimulation->ConstantTimeStep = ConstantTimeStep;
	ClothSimulation->WindForce = WindForce;
	ClothSimulation->GenerateCloth();

	if (bUseConstantTimeStep)
	{
		ClothSimulation->StartConstantTimer();
	}

	GenerateMesh();
}

void AClothActor::GenerateGrid()
{
	int32 VertexIndex = 0;
	for (int32 Y = 0; Y < GridSize; ++Y)
	{
		for (int32 X = 0; X < GridSize; ++X, ++VertexIndex)
		{
			// Each vertex gets a position from the simulation and a texture coordinate.
			FVector Position = ClothSimulation->GetNodePosition(VertexIndex);
			MeshData.Positions.Add(Position);

			const float U = static_cast<float>(X) / GridSize;
			const float V = static_cast<float>(Y) / GridSize;
			MeshData.UV0.Add({U, V});

			// Add an empty tangent and normal to be filled later.
			MeshData.Tangents.Add(FVector::ZeroVector);
			MeshData.Normals.Add(FVector::ZeroVector);

			// And add white.
			MeshData.Colors.Add(FColor::White);

			// When we have enough vertices and are not at the start of a line, we can start
			// generating triangles from them. Each vertex adds 2 triangles to the grid (1 quad).
			if (X > 0 && Y > 0)
			{
				const int32 TopRightIndex = VertexIndex - GridSize;
				const int32 TopLeftIndex = VertexIndex - GridSize - 1;
				const int32 BottomLeftIndex = VertexIndex - 1;
				const int32 BottomRightIndex = VertexIndex;
				
				// Now create two triangles from these four vertices.
				// The order of these (clockwise/counter-clockwise) dictates which way the triangle will face.
				MeshData.Triangles.Add(BottomLeftIndex);
				MeshData.Triangles.Add(TopRightIndex);
				MeshData.Triangles.Add(TopLeftIndex);
				
				MeshData.Triangles.Add(BottomLeftIndex);
				MeshData.Triangles.Add(BottomRightIndex);
				MeshData.Triangles.Add(TopRightIndex);

				const FVector NormalCurrent = FVector::CrossProduct(
					MeshData.Positions[BottomLeftIndex] - MeshData.Positions[TopLeftIndex],
					MeshData.Positions[TopLeftIndex] - MeshData.Positions[TopRightIndex]).GetSafeNormal();
				
				MeshData.Normals[TopLeftIndex] = NormalCurrent;
				MeshData.Normals[TopRightIndex] = NormalCurrent;
				MeshData.Normals[BottomLeftIndex] = NormalCurrent;
				MeshData.Normals[BottomRightIndex] = NormalCurrent;

				// Add material indices for the two generated triangles
				MeshData.MaterialIndex.Add(0);
				MeshData.MaterialIndex.Add(0);
			}
		}
	}
}

void AClothActor::GenerateMesh()
{
	Mesh = RealtimeMeshComponent->InitializeRealtimeMesh<URealtimeMeshSimple>();
	Mesh->SetupMaterialSlot(0, TEXT("Cloth Material"), Material);

	MeshSectionKey = FRealtimeMeshSectionGroupKey::Create(0, TEXT(""));
	GenerateGrid();
	Mesh->CreateSectionGroup(MeshSectionKey, MeshData);
}

void AClothActor::UpdateMesh()
{
	int32 VertexIndex = 0;
	for (int32 Y = 0; Y < GridSize; ++Y)
	{
		for (int32 X = 0; X < GridSize; ++X, ++VertexIndex)
		{
			// Each vertex gets its position updated from the simulation.
			FVector Position = ClothSimulation->GetNodePosition(VertexIndex);
			MeshData.Positions[VertexIndex] = Position;

			// We also need to update normals for each of the triangles in the grid.
			if (X > 0 && Y > 0)
			{
				const int32 TopRightIndex = VertexIndex - GridSize;
				const int32 TopLeftIndex = VertexIndex - GridSize - 1;
				const int32 BottomLeftIndex = VertexIndex - 1;
				const int32 BottomRightIndex = VertexIndex;

				const FVector NormalCurrent = FVector::CrossProduct(
					MeshData.Positions[BottomLeftIndex] - MeshData.Positions[TopLeftIndex],
					MeshData.Positions[TopLeftIndex] - MeshData.Positions[TopRightIndex]).GetSafeNormal();

				MeshData.Tangents[TopLeftIndex] = NormalCurrent;
				MeshData.Tangents[TopRightIndex] = NormalCurrent;
				MeshData.Tangents[BottomLeftIndex] = NormalCurrent;
				MeshData.Tangents[BottomRightIndex] = NormalCurrent;
			}
		}
	}
	Mesh->UpdateSectionGroup(MeshSectionKey, MeshData);
}

// Called every frame
void AClothActor::Tick(float DeltaTime)
{
	Super::Tick(DeltaTime);
	
	ClothSimulation->TimeScale = TimeScale;

	UpdateMesh();
}

