#pragma once

#include "CoreMinimal.h"
#include "GameFramework/Actor.h"
#include "StamFluidBox.generated.h"

DECLARE_LOG_CATEGORY_EXTERN(StamFluidBox, All, All);

UCLASS()
class CVICENI5_API AStamFluidBox : public AActor
{
	GENERATED_BODY()
	
public:	
	// Sets default values for this actor's properties
	AStamFluidBox();

protected:
	// Called when the game starts or when spawned
	virtual void BeginPlay() override;

public:	
	// Called every frame
	virtual void Tick(float DeltaTime) override;

	UFUNCTION(BlueprintCallable, Category="Fluid Simulation")
	void AddFluidAtLocation(FVector WorldLocation, float Amount);

	UFUNCTION(BlueprintCallable, Category="Fluid Simulation")
	void AddForceAtLocation(FVector WorldLocation, FVector Force);

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="Fluid Simulation")
	bool bSimulating = true;
	
	UPROPERTY(EditAnywhere, BlueprintReadOnly, Category="Fluid Simulation|Parameters")
	FIntVector GridSize = {10, 10, 10};

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="Fluid Simulation|Parameters")
	bool bUseLinearSolver = true;
	
	UPROPERTY(EditAnywhere, Category="Fluid Simulation|Parameters")
	int LinearSolverSteps = 20;
	
	UPROPERTY(EditAnywhere, Category="Fluid Simulation|Parameters")
	float DiffusionFactor = 0.0f;
	
	UPROPERTY(EditAnywhere, Category="Fluid Simulation|Parameters")
	float Viscosity = 0.0f;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="Fluid Simulation|Display")
	FColor FluidColor = FColor::White;

	UPROPERTY(VisibleAnywhere, Category="Fluid Simulation|Display")
	TObjectPtr<UVolumeTexture> VolumeTexture;

	UPROPERTY(EditAnywhere, Category="Fluid Simulation|Display")
	TObjectPtr<UMaterial> Material;
	
private:
	enum BoundaryType
	{
		BT_None,
		BT_XWall,
		BT_YWall,
		BT_ZWall,
	};

	UPROPERTY()
	TObjectPtr<UStaticMeshComponent> StaticMeshComponent;

	// Variables holding the actual grid data. Each velocity field vector component is stored in a separate array.
	FIntVector DensityGridSize;
	TArray<float> Density, DensityPrev;
	TArray<float> VelocityX, VelocityXPrev;
	TArray<float> VelocityY, VelocityYPrev;
	TArray<float> VelocityZ, VelocityZPrev;

	
	// Creates a runtime VolumeTexture through Unreal API. This texture exists only when playing the game. It
	// has a single mip level with the full resolution of our simulation cell grid (minus the edge cells). 
	void CreateVolumeTransient();
	
	// Add time-scaled values from the source array Sources to the destination array Cells.
	void AddSources(TArray<float>& Cells, TArray<float>& Sources, float DeltaTime);
	
	// High-level part of the simulation which uses the above functions to compute velocity field changes in a
	// single step of the simulation.
	void VelocityStep(float DeltaTime);
	
	// High-level part of the simulation which uses the above functions to compute density changes in a single
	// step of the simulation.
	void DensityStep(float DeltaTime);

	
	// Diffuses the values in the source array. Density/velocity each cell has is shared with its neighbours to
	// simulate natural diffusion of gasses (and the force fields which affect them, i.e. wind). Used several times in
	// the simulation. This version computes what density each cell should have if diffusion happened in reverse. This
	// way each cell is a separate variable, resulting in a set of linear equations which can be solved using the
	// iterative Gauss-Seidel method.
	// BType Type of border limiting values to apply (used for velocities) - passed when calling SetBounds
	void DiffuseGood(BoundaryType BType, TArray<float>& X, TArray<float>& X0, float Diffusion, float DeltaTime);
	
	// Same as Diffuse, but calculates diffusion forward in time. The algorithm is more straightforward but is
	// very unstable, especially for larger diffusion factor and time step.
	// BType Type of border limiting values to apply (used for velocities) - passed when calling SetBounds
	void DiffuseBad(BoundaryType BType, TArray<float>& X, TArray<float>& X0, float Diffusion, float DeltaTime);
	
	// Computes the flow of the densities/forces due to advection from the vector field. This makes the fluid
	// move across the cells. Computation is done similarly to the Diffusion step - backwards in time. We look at a position
	// and linearly work out which cell the value came from. Since the resulting value is linearly interpolated from the
	// computed position's neighbouring cells, this step also does pseudo-diffusion.
	// BType Type of border limiting values to apply (used for velocities) - passed when calling SetBounds 
	void Advect(BoundaryType BType, TArray<float>& D, TArray<float>& D0, TArray<float>& VX, TArray<float>& VY, TArray<float>& VZ, float DeltaTime);
	
	// This step ensures conservation of mass in our fluid's velocity field. This is done by using Hodge decomposition.
	// First, a gradient is computed for the velocity field (into p), which is subtracted from our velocity field.
	// The result is a mass conserving field which is saved to the input velocity field arrays.
	void Project(TArray<float>& Vx, TArray<float>& Vy, TArray<float>& Vz, TArray<float>& Temp, TArray<float>& Divisions);
	
	
	// Applies edge conditions to the additional grid layer which is used in this simulation. This implementation
	// encloses the fluid in a solid box.
	// type Type of edge binding conditions to apply 
	void SetBounds(BoundaryType Type, TArray<float>& Data);


	// Converts a set of coordinates to a cell array index. Simplifies working with a single-dimensional array
	// representing a multi-dimensional structure.
	inline int CellIndex(int X, int Y, int Z) const;
	inline int CellIndex(FIntVector Pos) const;
};
